<?php
/**
 * This file is part of the LdapTools package.
 *
 * (c) Chad Sikorra <Chad.Sikorra@gmail.com>
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

namespace spec\LdapTools\Security;

use LdapTools\Connection\LdapConnectionInterface;
use LdapTools\DomainConfiguration;
use LdapTools\Object\LdapObject;
use LdapTools\Security\Ace\Ace;
use LdapTools\Security\Ace\AceFlags;
use LdapTools\Security\Ace\AceRights;
use LdapTools\Security\Ace\AceType;
use LdapTools\Security\Acl\Dacl;
use LdapTools\Security\SddlParser;
use LdapTools\Security\SecurityDescriptor;
use LdapTools\Security\SID;
use PhpSpec\ObjectBehavior;
use Prophecy\Argument;

class SddlParserSpec extends ObjectBehavior
{
    function let(LdapConnectionInterface $connection)
    {
        $connection->getConfig()->willReturn(new DomainConfiguration('sub.example.local'));
        $connection->getRootDse()->willReturn(new LdapObject([
            'rootDomainNamingContext' => 'example.local',
            'defaultNamingContext' => 'sub.example.local',
        ]));

        $this->beConstructedWith($connection);
    }

    function it_is_initializable()
    {
        $this->shouldHaveType(SddlParser::class);
    }

    function it_should_parse_a_SDDL_string_that_represents_the_default_exchange_mailbox_permissions()
    {
        $exchHex = '010004801400000020000000000000002c00000001010000000000050a00000001010000000000050a00000004001c0001000000000214000100020001010000000000050a000000';
        $exchSddl = 'O:PSG:PSD:(A;CI;RCCC;;;PS)';
        $this->parse($exchSddl)->toBinary()->shouldBeEqualTo(hex2bin($exchHex));

        $sd = (new SecurityDescriptor())->setOwner(new SID('PS'))->setGroup(new SID('PS'));
        $acl = new Dacl();
        $ace = (new Ace(new AceType('A')))
            ->setFlags((new AceFlags())->add(AceFlags::SHORT_NAME['CI']))
            ->setRights((new AceRights())->add(AceRights::SHORT_NAME['RC'])->add(AceRights::SHORT_NAME['CC']))
            ->setTrustee(new SID('PS'));
        $acl->addAce($ace);
        $sd->setDacl($acl);

        $this->parse($exchSddl)->toBinary()->shouldBeEqualTo($sd->toBinary());

    }

    function it_should_parse_a_SDDL_string_that_represents_a_typical_ntSecurityDescriptor($connection)
    {
        $ntSecBin = hex2bin('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');
        $sddl = "O:DAG:DAD:AI(OA;;RP;4c164200-20c0-11d0-a768-00aa006e0529;;RS)(OA;;RP;5f202010-79a5-11d0-9020-00c04fc2d4cf;;RS)(OA;;RP;bc0ac240-79a9-11d0-9020-00c04fc2d4cf;;RS)(OA;;RP;037088f8-0ae1-11d2-b422-00a0c968f939;;RS)(OA;;RPWP;bf967a7f-0de6-11d0-a285-00aa003049e2;;CA)(OA;;RP;46a9b11d-60ae-405a-b7e8-ff8a58d456d2;;S-1-5-32-560)(OA;;RPWP;6db69a1c-9422-11d1-aebd-0000f80367c1;;S-1-5-32-561)(OA;;RPWP;5805bc62-bdc9-4428-a5e2-856a0f4c185e;;S-1-5-32-561)(OA;;CR;ab721a53-1e2f-11d0-9819-00aa0040529b;;WD)(OA;;CR;ab721a53-1e2f-11d0-9819-00aa0040529b;;PS)(OA;;CR;ab721a54-1e2f-11d0-9819-00aa0040529b;;PS)(OA;;CR;ab721a56-1e2f-11d0-9819-00aa0040529b;;PS)(OA;;RP;59ba2f42-79a2-11d0-9020-00c04fc2d3cf;;AU)(OA;;RP;e48d0154-bcf8-11d1-8702-00c04fb96050;;AU)(OA;;RP;77b5b886-944a-11d1-aebd-0000f80367c1;;AU)(OA;;RP;e45795b3-9455-11d1-aebd-0000f80367c1;;AU)(OA;;RPWP;77b5b886-944a-11d1-aebd-0000f80367c1;;PS)(OA;;RPWP;e45795b2-9455-11d1-aebd-0000f80367c1;;PS)(OA;;RPWP;e45795b3-9455-11d1-aebd-0000f80367c1;;PS)(A;;DERCWDWOCCDCLCSWRPWPDTLOCR;;;DA)(A;;DERCWDWOCCDCLCSWRPWPDTLOCR;;;AO)(A;;RC;;;AU)(A;;RCLCRPLO;;;PS)(A;;DERCWDWOCCDCLCSWRPWPDTLOCR;;;SY)(OA;CIID;CR;ab721a53-1e2f-11d0-9819-00aa0040529b;bf967aba-0de6-11d0-a285-00aa003049e2;S-1-5-21-1263317781-1938881490-3107577794-1119)(OA;CIID;CR;00299570-246d-11d0-a768-00aa006e0529;bf967aba-0de6-11d0-a285-00aa003049e2;S-1-5-21-1263317781-1938881490-3107577794-1119)(OA;CIIOID;CCDCLC;c975c901-6cea-4b6f-8319-d67f45449506;4828cc14-1437-45bc-9b07-ad6f015e5f28;S-1-5-21-1263317781-1938881490-3107577794-1116)(OA;CIID;CCDCLC;c975c901-6cea-4b6f-8319-d67f45449506;bf967aba-0de6-11d0-a285-00aa003049e2;S-1-5-21-1263317781-1938881490-3107577794-1116)(OA;CIIOID;RP;4c164200-20c0-11d0-a768-00aa006e0529;4828cc14-1437-45bc-9b07-ad6f015e5f28;RU)(OA;CIID;RP;4c164200-20c0-11d0-a768-00aa006e0529;bf967aba-0de6-11d0-a285-00aa003049e2;RU)(OA;CIIOID;RP;5f202010-79a5-11d0-9020-00c04fc2d4cf;4828cc14-1437-45bc-9b07-ad6f015e5f28;RU)(OA;CIID;RP;5f202010-79a5-11d0-9020-00c04fc2d4cf;bf967aba-0de6-11d0-a285-00aa003049e2;RU)(OA;CIIOID;RP;bc0ac240-79a9-11d0-9020-00c04fc2d4cf;4828cc14-1437-45bc-9b07-ad6f015e5f28;RU)(OA;CIID;RP;bc0ac240-79a9-11d0-9020-00c04fc2d4cf;bf967aba-0de6-11d0-a285-00aa003049e2;RU)(OA;CIIOID;RP;59ba2f42-79a2-11d0-9020-00c04fc2d3cf;4828cc14-1437-45bc-9b07-ad6f015e5f28;RU)(OA;CIID;RP;59ba2f42-79a2-11d0-9020-00c04fc2d3cf;bf967aba-0de6-11d0-a285-00aa003049e2;RU)(OA;CIIOID;RP;037088f8-0ae1-11d2-b422-00a0c968f939;4828cc14-1437-45bc-9b07-ad6f015e5f28;RU)(OA;CIID;RP;037088f8-0ae1-11d2-b422-00a0c968f939;bf967aba-0de6-11d0-a285-00aa003049e2;RU)(OA;CIID;CC;4828cc14-1437-45bc-9b07-ad6f015e5f28;;S-1-5-21-1263317781-1938881490-3107577794-1119)(OA;CIID;CC;bf967a86-0de6-11d0-a285-00aa003049e2;;S-1-5-21-1263317781-1938881490-3107577794-1119)(OA;CIID;CC;bf967a9c-0de6-11d0-a285-00aa003049e2;;S-1-5-21-1263317781-1938881490-3107577794-1119)(OA;CIID;CC;bf967aa5-0de6-11d0-a285-00aa003049e2;;S-1-5-21-1263317781-1938881490-3107577794-1119)(OA;CIID;CC;bf967aba-0de6-11d0-a285-00aa003049e2;;S-1-5-21-1263317781-1938881490-3107577794-1119)(OA;CIID;CC;5cb41ed0-0e4c-11d0-a286-00aa003049e2;;S-1-5-21-1263317781-1938881490-3107577794-1119)(OA;CIID;RP;4c164200-20c0-11d0-a768-00aa006e0529;;S-1-5-21-1263317781-1938881490-3107577794-1113)(OA;CIID;RP;b1b3a417-ec55-4191-b327-b72e33e38af2;;S-1-5-21-1263317781-1938881490-3107577794-1116)(OA;CIID;RP;9a7ad945-ca53-11d1-bbd0-0080c76670c0;;S-1-5-21-1263317781-1938881490-3107577794-1116)(OA;CIID;RP;bf967a68-0de6-11d0-a285-00aa003049e2;;S-1-5-21-1263317781-1938881490-3107577794-1116)(OA;CIID;RP;1f298a89-de98-47b8-b5cd-572ad53d267e;;S-1-5-21-1263317781-1938881490-3107577794-1116)(OA;CIID;RP;bf967991-0de6-11d0-a285-00aa003049e2;;S-1-5-21-1263317781-1938881490-3107577794-1116)(OA;CIID;RP;5fd424a1-1262-11d0-a060-00aa006c33ed;;S-1-5-21-1263317781-1938881490-3107577794-1116)(OA;CIID;WP;bf967a06-0de6-11d0-a285-00aa003049e2;;S-1-5-21-1263317781-1938881490-3107577794-1104)(OA;CIID;WP;bf967a06-0de6-11d0-a285-00aa003049e2;;S-1-5-21-1263317781-1938881490-3107577794-1117)(OA;CIID;WP;bf967a0a-0de6-11d0-a285-00aa003049e2;;S-1-5-21-1263317781-1938881490-3107577794-1119)(OA;CIID;WP;3e74f60e-3e73-11d1-a9c0-0000f80367c1;;S-1-5-21-1263317781-1938881490-3107577794-1104)(OA;CIID;WP;3e74f60e-3e73-11d1-a9c0-0000f80367c1;;S-1-5-21-1263317781-1938881490-3107577794-1117)(OA;CIID;WP;b1b3a417-ec55-4191-b327-b72e33e38af2;;S-1-5-21-1263317781-1938881490-3107577794-1104)(OA;CIID;WP;b1b3a417-ec55-4191-b327-b72e33e38af2;;S-1-5-21-1263317781-1938881490-3107577794-1117)(OA;CIID;WP;bf96791a-0de6-11d0-a285-00aa003049e2;;S-1-5-21-1263317781-1938881490-3107577794-1104)(OA;CIID;WP;bf96791a-0de6-11d0-a285-00aa003049e2;;S-1-5-21-1263317781-1938881490-3107577794-1117)(OA;CIID;WP;9a9a021e-4a5b-11d1-a9c3-0000f80367c1;;S-1-5-21-1263317781-1938881490-3107577794-1116)(OA;CIID;WP;0296c120-40da-11d1-a9c0-0000f80367c1;;S-1-5-21-1263317781-1938881490-3107577794-1119)(OA;CIID;WP;934de926-b09e-11d2-aa06-00c04f8eedd8;;S-1-5-21-1263317781-1938881490-3107577794-1116)(OA;CIID;WP;5e353847-f36c-48be-a7f7-49685402503c;;S-1-5-21-1263317781-1938881490-3107577794-1116)(OA;CIID;WP;8d3bca50-1d7e-11d0-a081-00aa006c33ed;;S-1-5-21-1263317781-1938881490-3107577794-1116)(OA;CIID;WP;bf967953-0de6-11d0-a285-00aa003049e2;;S-1-5-21-1263317781-1938881490-3107577794-1104)(OA;CIID;WP;bf967953-0de6-11d0-a285-00aa003049e2;;S-1-5-21-1263317781-1938881490-3107577794-1117)(OA;CIID;WP;e48d0154-bcf8-11d1-8702-00c04fb96050;;S-1-5-21-1263317781-1938881490-3107577794-1117)(OA;CIID;WP;275b2f54-982d-4dcd-b0ad-e53501445efb;;S-1-5-21-1263317781-1938881490-3107577794-1116)(OA;CIID;WP;bf967954-0de6-11d0-a285-00aa003049e2;;S-1-5-21-1263317781-1938881490-3107577794-1104)(OA;CIID;WP;bf967954-0de6-11d0-a285-00aa003049e2;;S-1-5-21-1263317781-1938881490-3107577794-1117)(OA;CIID;WP;bf967961-0de6-11d0-a285-00aa003049e2;;S-1-5-21-1263317781-1938881490-3107577794-1104)(OA;CIID;WP;bf967961-0de6-11d0-a285-00aa003049e2;;S-1-5-21-1263317781-1938881490-3107577794-1117)(OA;CIID;WP;bf967a68-0de6-11d0-a285-00aa003049e2;;S-1-5-21-1263317781-1938881490-3107577794-1119)(OA;CIID;WP;5fd42471-1262-11d0-a060-00aa006c33ed;;S-1-5-21-1263317781-1938881490-3107577794-1119)(OA;CIID;WP;5430e777-c3ea-4024-902e-dde192204669;;S-1-5-21-1263317781-1938881490-3107577794-1116)(OA;CIID;WP;6f606079-3a82-4c1b-8efb-dcc8c91d26fe;;S-1-5-21-1263317781-1938881490-3107577794-1116)(OA;CIID;WP;bf967a7a-0de6-11d0-a285-00aa003049e2;;S-1-5-21-1263317781-1938881490-3107577794-1119)(OA;CIID;WP;bf967a7f-0de6-11d0-a285-00aa003049e2;;S-1-5-21-1263317781-1938881490-3107577794-1116)(OA;CIID;WP;614aea82-abc6-4dd0-a148-d67a59c72816;;S-1-5-21-1263317781-1938881490-3107577794-1116)(OA;CIID;WP;66437984-c3c5-498f-b269-987819ef484b;;S-1-5-21-1263317781-1938881490-3107577794-1116)(OA;CIID;WP;77b5b886-944a-11d1-aebd-0000f80367c1;;S-1-5-21-1263317781-1938881490-3107577794-1117)(OA;CIID;WP;a8df7489-c5ea-11d1-bbcb-0080c76670c0;;S-1-5-21-1263317781-1938881490-3107577794-1104)(OA;CIID;WP;a8df7489-c5ea-11d1-bbcb-0080c76670c0;;S-1-5-21-1263317781-1938881490-3107577794-1117)(OA;CIID;WP;1f298a89-de98-47b8-b5cd-572ad53d267e;;S-1-5-21-1263317781-1938881490-3107577794-1104)(OA;CIID;WP;1f298a89-de98-47b8-b5cd-572ad53d267e;;S-1-5-21-1263317781-1938881490-3107577794-1117)(OA;CIID;WP;f0f8ff9a-1191-11d0-a060-00aa006c33ed;;S-1-5-21-1263317781-1938881490-3107577794-1104)(OA;CIID;WP;f0f8ff9a-1191-11d0-a060-00aa006c33ed;;S-1-5-21-1263317781-1938881490-3107577794-1116)(OA;CIID;WP;f0f8ff9a-1191-11d0-a060-00aa006c33ed;;S-1-5-21-1263317781-1938881490-3107577794-1117)(OA;CIID;WP;2cc06e9d-6f7e-426a-8825-0215de176e11;;S-1-5-21-1263317781-1938881490-3107577794-1116)(OA;CIID;WP;5fd424a1-1262-11d0-a060-00aa006c33ed;;S-1-5-21-1263317781-1938881490-3107577794-1104)(OA;CIID;WP;5fd424a1-1262-11d0-a060-00aa006c33ed;;S-1-5-21-1263317781-1938881490-3107577794-1117)(OA;CIID;WP;3263e3b8-fd6b-4c60-87f2-34bdaa9d69eb;;S-1-5-21-1263317781-1938881490-3107577794-1116)(OA;CIID;WP;28630ebc-41d5-11d1-a9c1-0000f80367c1;;S-1-5-21-1263317781-1938881490-3107577794-1104)(OA;CIID;WP;28630ebc-41d5-11d1-a9c1-0000f80367c1;;S-1-5-21-1263317781-1938881490-3107577794-1117)(OA;CIID;WP;bf9679c0-0de6-11d0-a285-00aa003049e2;;S-1-5-21-1263317781-1938881490-3107577794-1119)(OA;CIID;WP;3e0abfd0-126a-11d0-a060-00aa006c33ed;;S-1-5-21-1263317781-1938881490-3107577794-1119)(OA;CIID;WP;7cb4c7d3-8787-42b0-b438-3c5d479ad31e;;S-1-5-21-1263317781-1938881490-3107577794-1116)(OA;CIID;RPWP;5b47d60f-6090-40b2-9f37-2a4de88f3063;;S-1-5-21-1263317781-1938881490-3107577794-526)(OA;CIID;RPWP;5b47d60f-6090-40b2-9f37-2a4de88f3063;;S-1-5-21-1263317781-1938881490-3107577794-527)(OA;CIIOID;WDDT;;4828cc14-1437-45bc-9b07-ad6f015e5f28;S-1-5-21-1263317781-1938881490-3107577794-1119)(OA;CIID;WDDT;;bf967aba-0de6-11d0-a285-00aa003049e2;S-1-5-21-1263317781-1938881490-3107577794-1119)(OA;CIID;CCDCLCRPWPLO;f0f8ffac-1191-11d0-a060-00aa006c33ed;;S-1-5-21-1263317781-1938881490-3107577794-1117)(OA;CIID;CCDCLCRPWPLO;e8b2aff2-59a7-4eac-9a70-819adef701dd;;S-1-5-21-1263317781-1938881490-3107577794-1116)(OA;CIID;DERCWDWOCCDCLCSWRPWPDTLOCR;018849b0-a981-11d2-a9ff-00c04f8eedd8;;S-1-5-21-1263317781-1938881490-3107577794-1104)(OA;CIID;DERCWDWOCCDCLCSWRPWPDTLOCR;018849b0-a981-11d2-a9ff-00c04f8eedd8;;S-1-5-21-1263317781-1938881490-3107577794-1117)(OA;CIIOID;DE;;4828cc14-1437-45bc-9b07-ad6f015e5f28;S-1-5-21-1263317781-1938881490-3107577794-1119)(OA;CIIOID;DE;;bf967a86-0de6-11d0-a285-00aa003049e2;S-1-5-21-1263317781-1938881490-3107577794-1119)(OA;CIIOID;DE;;bf967a9c-0de6-11d0-a285-00aa003049e2;S-1-5-21-1263317781-1938881490-3107577794-1119)(OA;CIIOID;DE;;bf967aa5-0de6-11d0-a285-00aa003049e2;S-1-5-21-1263317781-1938881490-3107577794-1119)(OA;CIID;DE;;bf967aba-0de6-11d0-a285-00aa003049e2;S-1-5-21-1263317781-1938881490-3107577794-1119)(OA;CIIOID;DE;;5cb41ed0-0e4c-11d0-a286-00aa003049e2;S-1-5-21-1263317781-1938881490-3107577794-1119)(OA;CIIOID;WD;;bf967a9c-0de6-11d0-a285-00aa003049e2;S-1-5-21-1263317781-1938881490-3107577794-1117)(OA;CIIOID;SW;9b026da6-0d3c-465c-8bee-5199d7165cba;bf967a86-0de6-11d0-a285-00aa003049e2;CO)(OA;CIIOID;SW;9b026da6-0d3c-465c-8bee-5199d7165cba;bf967a86-0de6-11d0-a285-00aa003049e2;PS)(OA;CIIOID;RP;b7c69e6d-2cc7-11d2-854e-00a0c983f608;bf967a86-0de6-11d0-a285-00aa003049e2;ED)(OA;CIIOID;RP;b7c69e6d-2cc7-11d2-854e-00a0c983f608;bf967a9c-0de6-11d0-a285-00aa003049e2;ED)(OA;CIID;RP;b7c69e6d-2cc7-11d2-854e-00a0c983f608;bf967aba-0de6-11d0-a285-00aa003049e2;ED)(OA;CIIOID;WP;ea1b7b93-5e48-46d5-bc6c-4df4fda78a35;bf967a86-0de6-11d0-a285-00aa003049e2;PS)(OA;CIIOID;DERCWDWOCCDCLCSWRPWPDTLOCR;;c975c901-6cea-4b6f-8319-d67f45449506;S-1-5-21-1263317781-1938881490-3107577794-1117)(OA;CIIOID;DERCWDWOCCDCLCSWRPWPDTLOCR;;f0f8ffac-1191-11d0-a060-00aa006c33ed;S-1-5-21-1263317781-1938881490-3107577794-1117)(OA;CIIOID;RCLCRPLO;;4828cc14-1437-45bc-9b07-ad6f015e5f28;RU)(OA;CIIOID;RCLCRPLO;;bf967a9c-0de6-11d0-a285-00aa003049e2;RU)(OA;CIID;RCLCRPLO;;bf967aba-0de6-11d0-a285-00aa003049e2;RU)(OA;CIID;RP;b1b3a417-ec55-4191-b327-b72e33e38af2;;NS)(OA;CIID;RP;1f298a89-de98-47b8-b5cd-572ad53d267e;;AU)(OA;OICIID;RPWP;3f78c3e5-f79a-46bd-a0b8-9d18116ddc79;;PS)(OA;CIID;RPWPCR;91e647de-d96f-4b70-9557-d63ff4f3ccd8;;PS)(A;CIID;RCLCRPLO;;;S-1-5-21-1263317781-1938881490-3107577794-1104)(A;CIID;RCLCRPLO;;;S-1-5-21-1263317781-1938881490-3107577794-1117)(A;CIID;DERCWDWOCCDCLCSWRPWPDTLOCR;;;EA)(A;CIID;LC;;;RU)(A;CIID;DERCWDWOCCLCSWRPWPLOCR;;;BA)S:AI(OU;CIIOIDSA;WP;f30e3bbe-9ff0-11d1-b603-0000f80367c1;bf967aa5-0de6-11d0-a285-00aa003049e2;WD)(OU;CIIOIDSA;WP;f30e3bbf-9ff0-11d1-b603-0000f80367c1;bf967aa5-0de6-11d0-a285-00aa003049e2;WD)";
        $connection->execute(Argument::that(function($op) {
            return $op->getFilter() == "(&(objectClass=domain)(objectClass=domainDns))";
        }))->willReturn([
            "count" => 1,
            0 => [
                "objectSid" => [
                    "count" => 1,
                    0 => hex2bin('01040000000000051500000015b34c4bd2fb9073c2df39b9'),
                ],
                0 => "objectSid",
                "count" => 1,
                "dn" => "dc=example,dc=local",
            ]
        ]);

        $this->parse($sddl)->getOwner()->toString()->shouldBeEqualTo('S-1-5-21-1263317781-1938881490-3107577794-512');
        $this->parse($sddl)->getGroup()->toString()->shouldBeEqualTo('S-1-5-21-1263317781-1938881490-3107577794-512');
        $this->parse($sddl)->getSacl()->getAces()->shouldHaveCount(2);
        $this->parse($sddl)->getDacl()->getAces()->shouldHaveCount(133);
        $this->parse($sddl)->toBinary()->shouldBeEqualTo($ntSecBin);
    }

    function it_should_set_any_sacl_or_dacl_flags()
    {
        $this->parse('O:PSG:PSD:PARAI(A;CI;RCCC;;;PS)')->getControlFlags()->getValue()->shouldBeEqualTo(38148);
        $this->parse('O:PSG:PSS:PARAI(OU;CIIOIDSA;WP;f30e3bbf-9ff0-11d1-b603-0000f80367c1;bf967aa5-0de6-11d0-a285-00aa003049e2;WD)')->getControlFlags()->getValue()->shouldBeEqualTo(43536);
        $this->parse('O:PSG:PSD:P(A;CI;RCCC;;;PS)')->getControlFlags()->getValue()->shouldBeEqualTo(36868);
    }

    function it_should_throw_an_exception_if_the_SDDL_is_missing_the_owner_or_group()
    {
        $this->shouldThrow('LdapTools\Exception\SddlParserException')->duringParse('G:PSD:(A;CI;RCCC;;;PS)');
        $this->shouldThrow('LdapTools\Exception\SddlParserException')->duringParse('O:PSD:(A;CI;RCCC;;;PS)');
        $this->shouldThrow('LdapTools\Exception\SddlParserException')->duringParse('D:(A;CI;RCCC;;;PS)');
    }

    function it_should_throw_an_exception_if_the_owner_group_or_ace_sid_is_invalid()
    {
        $this->shouldThrow('LdapTools\Exception\SddlParserException')->duringParse('O:fooG:PSD:(A;CI;RCCC;;;PS)');
        $this->shouldThrow('LdapTools\Exception\SddlParserException')->duringParse('O:PSG:fooD:(A;CI;RCCC;;;PS)');
        $this->shouldThrow('LdapTools\Exception\SddlParserException')->duringParse('O:PSG:PSD:(A;CI;RCCC;;;foo)');
        $this->shouldThrow('LdapTools\Exception\SddlParserException')->duringParse('O:PSG:PSD:(A;CI;RCCC;;;)');
    }

    function it_should_throw_an_exception_if_the_DACL_or_SACL_appears_more_than_once_in_the_SDDL()
    {
        $this->shouldThrow('LdapTools\Exception\SddlParserException')->duringParse('O:PSG:PSD:(A;CI;RCCC;;;PS)D:(A;CI;RCCC;;;PS)');
        $this->shouldThrow('LdapTools\Exception\SddlParserException')->duringParse('O:PSG:PSS:(OU;CIIOIDSA;WP;f30e3bbf-9ff0-11d1-b603-0000f80367c1;bf967aa5-0de6-11d0-a285-00aa003049e2;WD)S:(OU;CIIOIDSA;WP;f30e3bbf-9ff0-11d1-b603-0000f80367c1;bf967aa5-0de6-11d0-a285-00aa003049e2;WD)');
    }

    function it_should_throw_an_exception_if_the_dacl_or_sacl_flags_are_invalid()
    {
        $this->shouldThrow('LdapTools\Exception\SddlParserException')->duringParse('O:PSG:PSD:FOO(A;CI;RCCC;;;PS)');
    }

    function it_should_throw_an_exception_if_invalid_flags_are_set_in_the_ACE()
    {
        $this->shouldThrow('LdapTools\Exception\SddlParserException')->duringParse('O:PSG:PSD:P(A;TT;RCCC;;;PS)');
    }
}
