<?php
/**
 * This file is part of the LdapTools package.
 *
 * (c) Chad Sikorra <Chad.Sikorra@gmail.com>
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

namespace spec\LdapTools\Security\Acl;

use LdapTools\Security\Ace\Ace;
use LdapTools\Security\Ace\AceFlags;
use LdapTools\Security\Acl\Acl;
use LdapTools\Security\Acl\Dacl;
use LdapTools\Security\SID;
use PhpSpec\ObjectBehavior;

class DaclSpec extends ObjectBehavior
{
    protected $expecteSddl = '(OA;CIIO;CCDCLC;c975c901-6cea-4b6f-8319-d67f45449506;4828cc14-1437-45bc-9b07-ad6f015e5f28;S-1-5-21-1263317781-1938881490-3107577794-1116)(OA;CIIO;CCDCLC;c975c901-6cea-4b6f-8319-d67f45449506;bf967aba-0de6-11d0-a285-00aa003049e2;S-1-5-21-1263317781-1938881490-3107577794-1116)(OA;;RP;4c164200-20c0-11d0-a768-00aa006e0529;4828cc14-1437-45bc-9b07-ad6f015e5f28;RU)(OA;;RP;4c164200-20c0-11d0-a768-00aa006e0529;bf967aba-0de6-11d0-a285-00aa003049e2;RU)(OA;;RP;5f202010-79a5-11d0-9020-00c04fc2d4cf;4828cc14-1437-45bc-9b07-ad6f015e5f28;RU)(OA;;RP;bc0ac240-79a9-11d0-9020-00c04fc2d4cf;4828cc14-1437-45bc-9b07-ad6f015e5f28;RU)(OA;;RP;bc0ac240-79a9-11d0-9020-00c04fc2d4cf;bf967aba-0de6-11d0-a285-00aa003049e2;RU)(OA;;RP;59ba2f42-79a2-11d0-9020-00c04fc2d3cf;4828cc14-1437-45bc-9b07-ad6f015e5f28;RU)(OA;;RP;59ba2f42-79a2-11d0-9020-00c04fc2d3cf;bf967aba-0de6-11d0-a285-00aa003049e2;RU)(OA;;RP;037088f8-0ae1-11d2-b422-00a0c968f939;4828cc14-1437-45bc-9b07-ad6f015e5f28;RU)(OA;;RP;037088f8-0ae1-11d2-b422-00a0c968f939;bf967aba-0de6-11d0-a285-00aa003049e2;RU)(OA;;CR;1131f6ab-9c07-11d1-f79f-00c04fc2dcd2;;S-1-5-21-1263317781-1938881490-3107577794-1116)(OA;;RPWP;bf967a7f-0de6-11d0-a285-00aa003049e2;;CA)(OA;CI;RP;4c164200-20c0-11d0-a768-00aa006e0529;;S-1-5-21-1263317781-1938881490-3107577794-1113)(OA;CI;RP;b1b3a417-ec55-4191-b327-b72e33e38af2;;S-1-5-21-1263317781-1938881490-3107577794-1116)(OA;CI;RP;9a7ad945-ca53-11d1-bbd0-0080c76670c0;;S-1-5-21-1263317781-1938881490-3107577794-1116)(OA;CI;RP;bf967a68-0de6-11d0-a285-00aa003049e2;;S-1-5-21-1263317781-1938881490-3107577794-1116)(OA;CI;RP;1f298a89-de98-47b8-b5cd-572ad53d267e;;S-1-5-21-1263317781-1938881490-3107577794-1116)(OA;CI;RP;bf967991-0de6-11d0-a285-00aa003049e2;;S-1-5-21-1263317781-1938881490-3107577794-1116)(OA;CI;RP;5fd424a1-1262-11d0-a060-00aa006c33ed;;S-1-5-21-1263317781-1938881490-3107577794-1116)(OA;CI;WP;bf967a06-0de6-11d0-a285-00aa003049e2;;S-1-5-21-1263317781-1938881490-3107577794-1104)(OA;CI;WP;bf967a06-0de6-11d0-a285-00aa003049e2;;S-1-5-21-1263317781-1938881490-3107577794-1117)(OA;CI;WP;3e74f60e-3e73-11d1-a9c0-0000f80367c1;;S-1-5-21-1263317781-1938881490-3107577794-1104)(OA;CI;WP;3e74f60e-3e73-11d1-a9c0-0000f80367c1;;S-1-5-21-1263317781-1938881490-3107577794-1117)(OA;CI;WP;b1b3a417-ec55-4191-b327-b72e33e38af2;;S-1-5-21-1263317781-1938881490-3107577794-1104)(OA;CI;WP;b1b3a417-ec55-4191-b327-b72e33e38af2;;S-1-5-21-1263317781-1938881490-3107577794-1117)(OA;CI;WP;bf96791a-0de6-11d0-a285-00aa003049e2;;S-1-5-21-1263317781-1938881490-3107577794-1104)(OA;CI;WP;bf96791a-0de6-11d0-a285-00aa003049e2;;S-1-5-21-1263317781-1938881490-3107577794-1117)(OA;CI;WP;9a9a021e-4a5b-11d1-a9c3-0000f80367c1;;S-1-5-21-1263317781-1938881490-3107577794-1116)(OA;CI;WP;934de926-b09e-11d2-aa06-00c04f8eedd8;;S-1-5-21-1263317781-1938881490-3107577794-1116)(OA;CI;WP;5e353847-f36c-48be-a7f7-49685402503c;;S-1-5-21-1263317781-1938881490-3107577794-1116)(OA;CI;WP;8d3bca50-1d7e-11d0-a081-00aa006c33ed;;S-1-5-21-1263317781-1938881490-3107577794-1116)(OA;CI;WP;bf967953-0de6-11d0-a285-00aa003049e2;;S-1-5-21-1263317781-1938881490-3107577794-1104)(OA;CI;WP;bf967953-0de6-11d0-a285-00aa003049e2;;S-1-5-21-1263317781-1938881490-3107577794-1117)(OA;CI;WP;e48d0154-bcf8-11d1-8702-00c04fb96050;;S-1-5-21-1263317781-1938881490-3107577794-1117)(OA;CI;WP;275b2f54-982d-4dcd-b0ad-e53501445efb;;S-1-5-21-1263317781-1938881490-3107577794-1116)(OA;CI;WP;bf967954-0de6-11d0-a285-00aa003049e2;;S-1-5-21-1263317781-1938881490-3107577794-1104)(OA;CI;WP;bf967954-0de6-11d0-a285-00aa003049e2;;S-1-5-21-1263317781-1938881490-3107577794-1117)(OA;CI;WP;bf967961-0de6-11d0-a285-00aa003049e2;;S-1-5-21-1263317781-1938881490-3107577794-1104)(OA;CI;WP;bf967961-0de6-11d0-a285-00aa003049e2;;S-1-5-21-1263317781-1938881490-3107577794-1117)(OA;CI;WP;5430e777-c3ea-4024-902e-dde192204669;;S-1-5-21-1263317781-1938881490-3107577794-1116)(OA;CI;WP;6f606079-3a82-4c1b-8efb-dcc8c91d26fe;;S-1-5-21-1263317781-1938881490-3107577794-1116)(OA;CI;WP;bf967a7f-0de6-11d0-a285-00aa003049e2;;S-1-5-21-1263317781-1938881490-3107577794-1116)(OA;CI;WP;614aea82-abc6-4dd0-a148-d67a59c72816;;S-1-5-21-1263317781-1938881490-3107577794-1116)(OA;CI;WP;66437984-c3c5-498f-b269-987819ef484b;;S-1-5-21-1263317781-1938881490-3107577794-1116)(OA;CI;WP;77b5b886-944a-11d1-aebd-0000f80367c1;;S-1-5-21-1263317781-1938881490-3107577794-1117)(OA;CI;WP;a8df7489-c5ea-11d1-bbcb-0080c76670c0;;S-1-5-21-1263317781-1938881490-3107577794-1104)(OA;CI;WP;a8df7489-c5ea-11d1-bbcb-0080c76670c0;;S-1-5-21-1263317781-1938881490-3107577794-1117)(OA;CI;WP;1f298a89-de98-47b8-b5cd-572ad53d267e;;S-1-5-21-1263317781-1938881490-3107577794-1104)(OA;CI;WP;1f298a89-de98-47b8-b5cd-572ad53d267e;;S-1-5-21-1263317781-1938881490-3107577794-1117)(OA;CI;WP;f0f8ff9a-1191-11d0-a060-00aa006c33ed;;S-1-5-21-1263317781-1938881490-3107577794-1104)(OA;CI;WP;f0f8ff9a-1191-11d0-a060-00aa006c33ed;;S-1-5-21-1263317781-1938881490-3107577794-1116)(OA;CI;WP;f0f8ff9a-1191-11d0-a060-00aa006c33ed;;S-1-5-21-1263317781-1938881490-3107577794-1117)(OA;CI;WP;2cc06e9d-6f7e-426a-8825-0215de176e11;;S-1-5-21-1263317781-1938881490-3107577794-1116)(OA;CI;WP;5fd424a1-1262-11d0-a060-00aa006c33ed;;S-1-5-21-1263317781-1938881490-3107577794-1104)(OA;CI;WP;5fd424a1-1262-11d0-a060-00aa006c33ed;;S-1-5-21-1263317781-1938881490-3107577794-1117)(OA;CI;WP;3263e3b8-fd6b-4c60-87f2-34bdaa9d69eb;;S-1-5-21-1263317781-1938881490-3107577794-1116)(OA;CI;WP;28630ebc-41d5-11d1-a9c1-0000f80367c1;;S-1-5-21-1263317781-1938881490-3107577794-1104)(OA;CI;WP;28630ebc-41d5-11d1-a9c1-0000f80367c1;;S-1-5-21-1263317781-1938881490-3107577794-1117)(OA;CI;WP;7cb4c7d3-8787-42b0-b438-3c5d479ad31e;;S-1-5-21-1263317781-1938881490-3107577794-1116)(OA;CI;CCDCLCRPWPLO;f0f8ffac-1191-11d0-a060-00aa006c33ed;;S-1-5-21-1263317781-1938881490-3107577794-1117)(OA;CI;CCDCLCRPWPLO;e8b2aff2-59a7-4eac-9a70-819adef701dd;;S-1-5-21-1263317781-1938881490-3107577794-1116)(OA;CI;DERCWDWOCCDCLCSWRPWPDTLOCR;018849b0-a981-11d2-a9ff-00c04f8eedd8;;S-1-5-21-1263317781-1938881490-3107577794-1104)(OA;CI;DERCWDWOCCDCLCSWRPWPDTLOCR;018849b0-a981-11d2-a9ff-00c04f8eedd8;;S-1-5-21-1263317781-1938881490-3107577794-1117)(OA;CIIO;WD;;bf967a9c-0de6-11d0-a285-00aa003049e2;S-1-5-21-1263317781-1938881490-3107577794-1117)(OA;CIIO;DERCWDWOCCDCLCSWRPWPDTLOCR;;c975c901-6cea-4b6f-8319-d67f45449506;S-1-5-21-1263317781-1938881490-3107577794-1117)(OA;CIIO;DERCWDWOCCDCLCSWRPWPDTLOCR;;f0f8ffac-1191-11d0-a060-00aa006c33ed;S-1-5-21-1263317781-1938881490-3107577794-1117)(OA;CINPIO;DERPWPLO;;e8b2aff2-59a7-4eac-9a70-819adef701dd;S-1-5-21-1263317781-1938881490-3107577794-1116)(OA;;RP;46a9b11d-60ae-405a-b7e8-ff8a58d456d2;;S-1-5-32-560)(OA;;RPWP;6db69a1c-9422-11d1-aebd-0000f80367c1;;S-1-5-32-561)(OA;;RPWP;5805bc62-bdc9-4428-a5e2-856a0f4c185e;;S-1-5-32-561)(OA;;RCLCRPLO;;4828cc14-1437-45bc-9b07-ad6f015e5f28;RU)(OA;;RCLCRPLO;;bf967aba-0de6-11d0-a285-00aa003049e2;RU)(OA;;CR;ab721a53-1e2f-11d0-9819-00aa0040529b;;WD)(OA;;CR;ab721a53-1e2f-11d0-9819-00aa0040529b;;PS)(OA;CI;RP;b1b3a417-ec55-4191-b327-b72e33e38af2;;NS)(OA;CI;RP;1f298a89-de98-47b8-b5cd-572ad53d267e;;AU)(OA;CI;RPWPCR;91e647de-d96f-4b70-9557-d63ff4f3ccd8;;PS)(A;;RCWDWOCCDCLCSWRPWPLOCR;;;DA)(A;;RCWDWOCCDCLCSWRPWPLOCR;;;EA)(A;CI;RCLCRPLO;;;S-1-5-21-1263317781-1938881490-3107577794-1104)(A;CI;RCLCRPLO;;;S-1-5-21-1263317781-1938881490-3107577794-1117)(A;;DERCWDWOCCDCLCSWRPWPLOCR;;;BA)(A;;RCLCRPLO;;;AU)(A;;DERCWDWOCCDCLCSWRPWPDTLOCR;;;SY)';

    function let()
    {
        $dacl = "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";

        $this->beConstructedWith(hex2bin($dacl));
    }

    function it_is_initializable()
    {
        $this->shouldHaveType(Dacl::class);
    }

    function it_should_parse_and_get_the_ACL_revision()
    {
        $this->getRevision()->shouldBeEqualTo(4);
    }

    function it_should_parse_and_get_the_sbz1_value()
    {
        $this->getSbz1()->shouldBeEqualTo(0);
    }

    function it_should_set_the_sbz1_value()
    {
        $this->setSbz1(1)->getSbz1()->shouldBeEqualTo(1);
    }

    function it_should_parse_and_get_the_sbz2_value()
    {
        $this->getSbz2()->shouldBeEqualTo(0);
    }

    function it_should_set_the_sbz2_value()
    {
        $this->setSbz2(1)->getSbz2()->shouldBeEqualTo(1);
    }

    function it_should_parse_and_get_the_ACEs_for_the_ACL()
    {
        $this->getAces()->shouldBeArray();
        $this->getAces()->shouldHaveCount(85);
    }

    function it_should_get_the_binary_representation()
    {
        $dacl = "0400a01155000000050a4800070000000300000001c975c9ea6c6f4b8319d67f4544950614cc28483714bc459b07ad6f015e5f2801050000000000051500000015b34c4bd2fb9073c2df39b95c040000050a4800070000000300000001c975c9ea6c6f4b8319d67f45449506ba7a96bfe60dd011a28500aa003049e201050000000000051500000015b34c4bd2fb9073c2df39b95c04000005003c0010000000030000000042164cc020d011a76800aa006e052914cc28483714bc459b07ad6f015e5f280102000000000005200000002a02000005003c0010000000030000000042164cc020d011a76800aa006e0529ba7a96bfe60dd011a28500aa003049e20102000000000005200000002a02000005003c0010000000030000001020205fa579d011902000c04fc2d4cf14cc28483714bc459b07ad6f015e5f280102000000000005200000002a02000005003c00100000000300000040c20abca979d011902000c04fc2d4cf14cc28483714bc459b07ad6f015e5f280102000000000005200000002a02000005003c00100000000300000040c20abca979d011902000c04fc2d4cfba7a96bfe60dd011a28500aa003049e20102000000000005200000002a02000005003c001000000003000000422fba59a279d011902000c04fc2d3cf14cc28483714bc459b07ad6f015e5f280102000000000005200000002a02000005003c001000000003000000422fba59a279d011902000c04fc2d3cfba7a96bfe60dd011a28500aa003049e20102000000000005200000002a02000005003c001000000003000000f8887003e10ad211b42200a0c968f93914cc28483714bc459b07ad6f015e5f280102000000000005200000002a02000005003c001000000003000000f8887003e10ad211b42200a0c968f939ba7a96bfe60dd011a28500aa003049e20102000000000005200000002a020000050038000001000001000000abf63111079cd111f79f00c04fc2dcd201050000000000051500000015b34c4bd2fb9073c2df39b95c0400000500380030000000010000007f7a96bfe60dd011a28500aa003049e201050000000000051500000015b34c4bd2fb9073c2df39b9050200000502380010000000010000000042164cc020d011a76800aa006e052901050000000000051500000015b34c4bd2fb9073c2df39b95904000005023800100000000100000017a4b3b155ec9141b327b72e33e38af201050000000000051500000015b34c4bd2fb9073c2df39b95c04000005023800100000000100000045d97a9a53cad111bbd00080c76670c001050000000000051500000015b34c4bd2fb9073c2df39b95c040000050238001000000001000000687a96bfe60dd011a28500aa003049e201050000000000051500000015b34c4bd2fb9073c2df39b95c040000050238001000000001000000898a291f98deb847b5cd572ad53d267e01050000000000051500000015b34c4bd2fb9073c2df39b95c040000050238001000000001000000917996bfe60dd011a28500aa003049e201050000000000051500000015b34c4bd2fb9073c2df39b95c040000050238001000000001000000a124d45f6212d011a06000aa006c33ed01050000000000051500000015b34c4bd2fb9073c2df39b95c040000050238002000000001000000067a96bfe60dd011a28500aa003049e201050000000000051500000015b34c4bd2fb9073c2df39b950040000050238002000000001000000067a96bfe60dd011a28500aa003049e201050000000000051500000015b34c4bd2fb9073c2df39b95d0400000502380020000000010000000ef6743e733ed111a9c00000f80367c101050000000000051500000015b34c4bd2fb9073c2df39b9500400000502380020000000010000000ef6743e733ed111a9c00000f80367c101050000000000051500000015b34c4bd2fb9073c2df39b95d04000005023800200000000100000017a4b3b155ec9141b327b72e33e38af201050000000000051500000015b34c4bd2fb9073c2df39b95004000005023800200000000100000017a4b3b155ec9141b327b72e33e38af201050000000000051500000015b34c4bd2fb9073c2df39b95d0400000502380020000000010000001a7996bfe60dd011a28500aa003049e201050000000000051500000015b34c4bd2fb9073c2df39b9500400000502380020000000010000001a7996bfe60dd011a28500aa003049e201050000000000051500000015b34c4bd2fb9073c2df39b95d0400000502380020000000010000001e029a9a5b4ad111a9c30000f80367c101050000000000051500000015b34c4bd2fb9073c2df39b95c04000005023800200000000100000026e94d939eb0d211aa0600c04f8eedd801050000000000051500000015b34c4bd2fb9073c2df39b95c0400000502380020000000010000004738355e6cf3be48a7f749685402503c01050000000000051500000015b34c4bd2fb9073c2df39b95c04000005023800200000000100000050ca3b8d7e1dd011a08100aa006c33ed01050000000000051500000015b34c4bd2fb9073c2df39b95c040000050238002000000001000000537996bfe60dd011a28500aa003049e201050000000000051500000015b34c4bd2fb9073c2df39b950040000050238002000000001000000537996bfe60dd011a28500aa003049e201050000000000051500000015b34c4bd2fb9073c2df39b95d04000005023800200000000100000054018de4f8bcd111870200c04fb9605001050000000000051500000015b34c4bd2fb9073c2df39b95d040000050238002000000001000000542f5b272d98cd4db0ade53501445efb01050000000000051500000015b34c4bd2fb9073c2df39b95c040000050238002000000001000000547996bfe60dd011a28500aa003049e201050000000000051500000015b34c4bd2fb9073c2df39b950040000050238002000000001000000547996bfe60dd011a28500aa003049e201050000000000051500000015b34c4bd2fb9073c2df39b95d040000050238002000000001000000617996bfe60dd011a28500aa003049e201050000000000051500000015b34c4bd2fb9073c2df39b950040000050238002000000001000000617996bfe60dd011a28500aa003049e201050000000000051500000015b34c4bd2fb9073c2df39b95d04000005023800200000000100000077e73054eac32440902edde19220466901050000000000051500000015b34c4bd2fb9073c2df39b95c0400000502380020000000010000007960606f823a1b4c8efbdcc8c91d26fe01050000000000051500000015b34c4bd2fb9073c2df39b95c0400000502380020000000010000007f7a96bfe60dd011a28500aa003049e201050000000000051500000015b34c4bd2fb9073c2df39b95c04000005023800200000000100000082ea4a61c6abd04da148d67a59c7281601050000000000051500000015b34c4bd2fb9073c2df39b95c04000005023800200000000100000084794366c5c38f49b269987819ef484b01050000000000051500000015b34c4bd2fb9073c2df39b95c04000005023800200000000100000086b8b5774a94d111aebd0000f80367c101050000000000051500000015b34c4bd2fb9073c2df39b95d0400000502380020000000010000008974dfa8eac5d111bbcb0080c76670c001050000000000051500000015b34c4bd2fb9073c2df39b9500400000502380020000000010000008974dfa8eac5d111bbcb0080c76670c001050000000000051500000015b34c4bd2fb9073c2df39b95d040000050238002000000001000000898a291f98deb847b5cd572ad53d267e01050000000000051500000015b34c4bd2fb9073c2df39b950040000050238002000000001000000898a291f98deb847b5cd572ad53d267e01050000000000051500000015b34c4bd2fb9073c2df39b95d0400000502380020000000010000009afff8f09111d011a06000aa006c33ed01050000000000051500000015b34c4bd2fb9073c2df39b9500400000502380020000000010000009afff8f09111d011a06000aa006c33ed01050000000000051500000015b34c4bd2fb9073c2df39b95c0400000502380020000000010000009afff8f09111d011a06000aa006c33ed01050000000000051500000015b34c4bd2fb9073c2df39b95d0400000502380020000000010000009d6ec02c7e6f6a4288250215de176e1101050000000000051500000015b34c4bd2fb9073c2df39b95c040000050238002000000001000000a124d45f6212d011a06000aa006c33ed01050000000000051500000015b34c4bd2fb9073c2df39b950040000050238002000000001000000a124d45f6212d011a06000aa006c33ed01050000000000051500000015b34c4bd2fb9073c2df39b95d040000050238002000000001000000b8e363326bfd604c87f234bdaa9d69eb01050000000000051500000015b34c4bd2fb9073c2df39b95c040000050238002000000001000000bc0e6328d541d111a9c10000f80367c101050000000000051500000015b34c4bd2fb9073c2df39b950040000050238002000000001000000bc0e6328d541d111a9c10000f80367c101050000000000051500000015b34c4bd2fb9073c2df39b95d040000050238002000000001000000d3c7b47c8787b042b4383c5d479ad31e01050000000000051500000015b34c4bd2fb9073c2df39b95c04000005023800b700000001000000acfff8f09111d011a06000aa006c33ed01050000000000051500000015b34c4bd2fb9073c2df39b95d04000005023800b700000001000000f2afb2e8a759ac4e9a70819adef701dd01050000000000051500000015b34c4bd2fb9073c2df39b95c04000005023800ff010f0001000000b049880181a9d211a9ff00c04f8eedd801050000000000051500000015b34c4bd2fb9073c2df39b95004000005023800ff010f0001000000b049880181a9d211a9ff00c04f8eedd801050000000000051500000015b34c4bd2fb9073c2df39b95d040000050a380000000400020000009c7a96bfe60dd011a28500aa003049e201050000000000051500000015b34c4bd2fb9073c2df39b95d040000050a3800ff010f000200000001c975c9ea6c6f4b8319d67f4544950601050000000000051500000015b34c4bd2fb9073c2df39b95d040000050a3800ff010f0002000000acfff8f09111d011a06000aa006c33ed01050000000000051500000015b34c4bd2fb9073c2df39b95d040000050e3800b000010002000000f2afb2e8a759ac4e9a70819adef701dd01050000000000051500000015b34c4bd2fb9073c2df39b95c04000005002c0010000000010000001db1a946ae605a40b7e8ff8a58d456d20102000000000005200000003002000005002c0030000000010000001c9ab66d2294d111aebd0000f80367c10102000000000005200000003102000005002c00300000000100000062bc0558c9bd2844a5e2856a0f4c185e0102000000000005200000003102000005002c00940002000200000014cc28483714bc459b07ad6f015e5f280102000000000005200000002a02000005002c009400020002000000ba7a96bfe60dd011a28500aa003049e20102000000000005200000002a020000050028000001000001000000531a72ab2f1ed011981900aa0040529b010100000000000100000000050028000001000001000000531a72ab2f1ed011981900aa0040529b01010000000000050a00000005022800100000000100000017a4b3b155ec9141b327b72e33e38af2010100000000000514000000050228001000000001000000898a291f98deb847b5cd572ad53d267e01010000000000050b000000050228003001000001000000de47e6916fd9704b9557d63ff4f3ccd801010000000000050a00000000002400bf010e0001050000000000051500000015b34c4bd2fb9073c2df39b90002000000002400bf010e0001050000000000051500000015b34c4bd2fb9073c2df39b907020000000224009400020001050000000000051500000015b34c4bd2fb9073c2df39b950040000000224009400020001050000000000051500000015b34c4bd2fb9073c2df39b95d04000000001800bf010f0001020000000000052000000020020000000014009400020001010000000000050b00000000001400ff010f00010100000000000512000000";

        $this->toBinary()->shouldBeEqualTo(hex2bin($dacl));
    }

    function it_should_set_the_ACL_revision()
    {
        $this->setRevision(Acl::REVISION['GENERIC'])->getRevision()->shouldBeEqualTo(Acl::REVISION['GENERIC']);
    }

    function it_should_get_the_sddl_representation_when_calling_toSddl()
    {
        $this->toSddl()->shouldBeEqualTo($this->expecteSddl);
    }

    function it_should_have_a_string_representation_that_outputs_the_sddl()
    {
        $this->__toString()->shouldBeEqualTo($this->expecteSddl);
    }

    function it_should_add_an_ace()
    {
        $ace = (new Ace('D'))->setTrustee(new SID('PS'));;

        $this->getAces()->shouldNotContain($ace);
        $this->addAce($ace)->getAces()->shouldContain($ace);
    }

    function it_should_remove_an_ace()
    {
        $ace = (new Ace('D'))->setTrustee(new SID('PS'));

        $this->addAce($ace)->getAces()->shouldContain($ace);
        $this->removeAce($ace)->getAces()->shouldNotContain($ace);
    }

    function it_should_tell_whether_the_ACEs_are_canonical_when_calling_isCanonical()
    {
        $this->isCanonical()->shouldBeEqualTo(true);
        $this->addAce((new Ace('D'))->setTrustee(new SID('PS')))->isCanonical()->shouldBeEqualTo(false);
    }

    function it_should_order_the_ace_canonically_if_specified()
    {
        $this->beConstructedWith(null);

        $deny = (new Ace('D'))->setTrustee(new SID('PS'));;
        $allow = (new Ace('A'))->setTrustee(new SID('PS'));;
        $denyObj = (new Ace('OD'))->setTrustee(new SID('PS'));;
        $allowObj = (new Ace('OA'))->setTrustee(new SID('PS'));;
        $inherited = (new Ace('D'))->setTrustee(new SID('PS'));;
        $inherited->getFlags()->add(AceFlags::FLAG['INHERITED']);
        $this->addAce($inherited, $allowObj, $allow, $deny, $denyObj);

        $this->canonicalize()->getAces()->shouldBeEqualTo([
            $denyObj,
            $deny,
            $allowObj,
            $allow,
            $inherited
        ]);
    }

    function it_should_order_the_aces_non_canonically_to_sddl_if_specified()
    {
        $this->beConstructedWith(null);
        $deny = (new Ace('D'))->setTrustee(new SID('PS'));
        $allow = (new Ace('A'))->setTrustee(new SID('PS'));;
        $this->addAce($allow, $deny);

        $this->toSddl(false)->shouldBeEqualTo('(A;;;;;PS)(D;;;;;PS)');
        $this->toSddl()->shouldBeEqualTo('(D;;;;;PS)(A;;;;;PS)');
    }

    function it_should_order_the_aces_non_canonically_to_binary_if_specified()
    {
        $this->beConstructedWith(null);
        $deny = (new Ace('D'))->setTrustee(new SID('PS'));
        $allow = (new Ace('A'))->setTrustee(new SID('PS'));;
        $this->addAce($allow, $deny);

        $this->toBinary(false)->shouldBeEqualTo(hex2bin('0400300002000000000014000000000001010000000000050a000000010014000000000001010000000000050a000000'));
        $this->toBinary()->shouldBeEqualTo(hex2bin('0400300002000000010014000000000001010000000000050a000000000014000000000001010000000000050a000000'));
    }

    function it_should_check_if_the_ace_exists()
    {
        $deny = (new Ace('D'))->setTrustee(new SID('PS'));
        $deny2 = (new Ace('D'))->setTrustee(new SID('PS'));
        $deny3 = (new Ace('D'))->setTrustee(new SID('AO'));
        $this->addAce($deny, $deny3);

        $this->hasAce($deny)->shouldBeEqualTo(true);
        $this->hasAce($deny2)->shouldBeEqualTo(false);
        $this->hasAce($deny, $deny2)->shouldBeEqualTo(false);
        $this->hasAce($deny, $deny3)->shouldBeEqualTo(true);
        $this->hasAce(...[])->shouldBeEqualTo(false);
    }
}
