<?php
require_once 'class.DTW.php';


class Test_DTW extends PHPUnit_Framework_TestCase {
    private $dtw;
    public function setUp() {
        $this->dtw = new DTW();
    }

    public function testIsClassExists() {
        $this->assertInstanceOf('DTW', $this->dtw);
    }

    public function testIsMethodsExists() {
        $this->assertTrue(method_exists($this->dtw, 'setReference'));
        $this->assertTrue(method_exists($this->dtw, 'setCheck'));
        $this->assertTrue(method_exists($this->dtw, 'getReference'));
        $this->assertTrue(method_exists($this->dtw, 'getCheck'));
        $this->assertTrue(method_exists($this->dtw, 'getMinCost'));
        $this->assertTrue(method_exists($this->dtw, 'd'));
        $this->assertTrue(method_exists($this->dtw, 'getMinPath'));
    }

    /**
     * @dataProvider provider
     */
    public function testReferenceSets($array) {
        $this->dtw->setReference($array);
        $this->assertEquals($this->dtw->getReference(), $array);
    }

    /**
     * @dataProvider provider
     */
    public function testCheckSets($array) {
        $this->dtw->setCheck($array);
        $this->assertEquals($this->dtw->getCheck(), $array);
    }

    /**
     * @dataProvider provider
     */
    public function testSets($array) {
        $this->dtw->setCheck($array)
                    ->setReference($array);
        $this->assertEquals($this->dtw->getMinCost(), 0);
    }

    public function testTwoSequences() {
        $this->dtw->setCheck(Array( Array(-2, 0), Array(10, 0), Array(-10, 0), Array(15, 0), Array(-13, 0), Array(20, 0), Array(-5, 0), Array(14, 0), Array(2,0)))
                    ->setReference(Array( Array(3, 0), Array(-13, 0), Array(14, 0), Array(-7, 0), Array(9, 0), Array(-2,0) ));
        $this->assertTrue($this->dtw->getMinCost() == 54/15);
    }

    public function testCount() {
        $check = Array(Array(0,0), Array(0,1.5), Array(0,1.5));
        $reference = Array(Array(0,0), Array(0,1), Array(0.5,1.5));
        $this->dtw->setCheck($check)
            ->setReference($reference);
        $this->assertTrue($this->dtw->getMinCost() == 1/6);
        $check = Array(Array(0,0), Array(1,1), Array(0,2));
        $reference = Array(Array(0,0), Array(-1,1), Array(0,2));
        $this->dtw->setCheck($check)
            ->setReference($reference);
        $this->assertTrue($this->dtw->getMinCost() == 2/6);
    }

    public function testCanvasData() {
        $reference = Array( Array(52, 25), Array(58, 29), Array(61, 31), Array(65, 33), Array(76, 37), Array(80, 37));
        $this->dtw->setReference($reference);
        $check = Array( Array(52, 25), Array(59, 30), Array(62, 30), Array(66, 37), Array(76, 37), Array(80, 37));
        $this->dtw->setCheck($check);
        $this->assertTrue($this->dtw->getMinCost() > 0);
    }

    public function testCanvasData2() {
        $reference = Array( Array(147, 57), Array(147, 56), Array(146, 56), Array(146, 55), Array(145, 55), Array(144, 55), Array(143, 55), Array(143, 54), Array(141, 54), Array(139, 53), Array(137, 53), Array(135, 53), Array(134, 53), Array(133, 53), Array(131, 53), Array(127, 53), Array(120, 53), Array(119, 53), Array(116, 53), Array(114, 53), Array(111, 57), Array(107, 62), Array(107, 67), Array(107, 73), Array(108, 77), Array(112, 82), Array(117, 84), Array(124, 88), Array(132, 89), Array(140, 90), Array(151, 90), Array(157, 90), Array(159, 89), Array(164, 81), Array(166, 79), Array(168, 75), Array(169, 73), Array(169, 70), Array(171, 67), Array(172, 64), Array(173, 58), Array(173, 48), Array(173, 45), Array(173, 42), Array(171, 33), Array(164, 27), Array(162, 27), Array(161, 27), Array(142, 27), Array(139, 27), Array(135, 27), Array(125, 27), Array(116, 27), Array(112, 27), Array(110, 27), Array(109, 28) );
        $this->dtw->setReference($reference);
        $check = Array( Array(160, 57), Array(159, 57), Array(158, 57), Array(158, 56), Array(157, 56), Array(156, 55), Array(155, 55), Array(152, 55), Array(151, 55), Array(150, 55), Array(149, 55), Array(148, 55), Array(147, 55), Array(142, 55), Array(140, 55), Array(140, 56), Array(139, 56), Array(137, 58), Array(135, 59), Array(133, 62), Array(132, 66), Array(131, 69), Array(131, 72), Array(131, 76), Array(133, 79), Array(135, 80), Array(136, 81), Array(137, 82), Array(140, 83), Array(142, 85), Array(146, 88), Array(152, 89), Array(154, 89), Array(155, 89), Array(159, 89), Array(160, 89), Array(165, 89), Array(167, 89), Array(168, 89), Array(171, 89), Array(173, 87), Array(174, 87), Array(179, 83), Array(180, 82), Array(183, 76), Array(185, 70), Array(186, 62), Array(186, 61), Array(186, 57), Array(186, 56), Array(186, 51), Array(186, 50), Array(185, 46), Array(185, 44), Array(184, 44), Array(183, 42), Array(182, 41), Array(181, 40), Array(179, 39), Array(178, 38), Array(177, 38), Array(174, 36), Array(171, 35), Array(169, 34), Array(167, 33), Array(165, 32), Array(163, 30), Array(162, 30), Array(159, 28), Array(154, 28), Array(150, 28), Array(142, 27), Array(139, 27), Array(137, 27), Array(136, 27) );
        $this->dtw->setCheck($check);
        $minCost = $this->dtw->getMinCost();
        $this->assertTrue($minCost > 0);

        $check = Array( Array(156, 69), Array(155, 69), Array(154, 69), Array(153, 69), Array(152, 69), Array(151, 69), Array(150, 69), Array(149, 69), Array(147, 69), Array(143, 70), Array(141, 70), Array(137, 72), Array(134, 74), Array(130, 81), Array(129, 83), Array(128, 85), Array(128, 86), Array(128, 93), Array(128, 95), Array(128, 96), Array(128, 97), Array(130, 101), Array(134, 103), Array(140, 106), Array(149, 106), Array(152, 106), Array(159, 106), Array(169, 104), Array(172, 103), Array(178, 99), Array(179, 97), Array(184, 94), Array(188, 89), Array(191, 78), Array(191, 76), Array(189, 66), Array(188, 64), Array(186, 62), Array(183, 55), Array(169, 42), Array(164, 40), Array(161, 39), Array(157, 38), Array(151, 38), Array(144, 38), Array(138, 40), Array(134, 43), Array(133, 44), Array(132, 46), Array(130, 46) );
        $this->dtw->setCheck($check);
        $minCost = $this->dtw->getMinCost();
        $this->assertTrue($minCost > 0);

        $check = Array( Array(168, 72), Array(167, 70), Array(166, 70), Array(165, 69), Array(164, 68), Array(162, 68), Array(162, 67), Array(161, 67), Array(160, 66), Array(159, 66), Array(157, 66), Array(156, 66), Array(155, 66), Array(151, 66), Array(145, 69), Array(138, 73), Array(136, 76), Array(135, 79), Array(133, 84), Array(130, 92), Array(133, 101), Array(134, 101), Array(137, 103), Array(140, 105), Array(149, 106), Array(157, 106), Array(159, 106), Array(167, 106), Array(176, 102), Array(180, 97), Array(183, 91), Array(186, 82), Array(186, 75), Array(187, 74), Array(187, 69), Array(187, 68), Array(187, 63), Array(186, 61), Array(179, 54), Array(173, 50), Array(164, 46), Array(157, 42), Array(156, 42), Array(154, 42), Array(152, 41), Array(150, 39), Array(149, 39), Array(143, 39), Array(142, 39), Array(137, 39), Array(136, 39), Array(134, 39), Array(133, 39), Array(133, 40), Array(131, 40), Array(130, 41) );
        $this->dtw->setCheck($check);
        $minCost = $this->dtw->getMinCost();
        $this->assertTrue($minCost > 0);

        $check = Array( Array(173, 64), Array(172, 64), Array(171, 64), Array(171, 63), Array(170, 63), Array(169, 63), Array(168, 63), Array(166, 62), Array(166, 61), Array(165, 61), Array(162, 61), Array(160, 61), Array(158, 61), Array(157, 61), Array(156, 61), Array(155, 61), Array(153, 61), Array(152, 62), Array(151, 63), Array(149, 63), Array(146, 65), Array(145, 65), Array(142, 67), Array(140, 71), Array(139, 73), Array(138, 77), Array(138, 81), Array(138, 85), Array(138, 86), Array(139, 86), Array(141, 88), Array(142, 89), Array(145, 91), Array(149, 92), Array(150, 93), Array(155, 93), Array(157, 93), Array(164, 94), Array(174, 92), Array(177, 91), Array(182, 87), Array(193, 79), Array(197, 70), Array(197, 60), Array(196, 55), Array(194, 48), Array(189, 44), Array(177, 37), Array(162, 33), Array(157, 33), Array(151, 32), Array(144, 32), Array(143, 33), Array(141, 33), Array(137, 35) );
        $this->dtw->setCheck($check);
        $minCost = $this->dtw->getMinCost();
        $this->assertTrue($minCost > 0);
    }

    public function testCanvasData3() {
        $reference = Array( Array(135, 139), Array(137, 138), Array(139, 138), Array(140, 137), Array(142, 135), Array(145, 135), Array(148, 134), Array(150, 133), Array(153, 132), Array(156, 131), Array(169, 129), Array(181, 125), Array(184, 124), Array(184, 122), Array(176, 118), Array(159, 111), Array(156, 111), Array(153, 110), Array(139, 105), Array(133, 102), Array(131, 100), Array(131, 97), Array(135, 94), Array(146, 90), Array(155, 86), Array(169, 81), Array(178, 77), Array(183, 70), Array(185, 66), Array(186, 61), Array(186, 59) );
        $this->dtw->setReference($reference);
        $check = Array( Array(123, 117), Array(124, 117), Array(125, 117), Array(126, 116), Array(128, 116), Array(130, 115), Array(132, 113), Array(137, 111), Array(142, 109), Array(161, 101), Array(165, 99), Array(168, 96), Array(169, 92), Array(168, 88), Array(164, 85), Array(157, 83), Array(150, 81), Array(145, 79), Array(137, 74), Array(130, 71), Array(124, 67), Array(122, 66), Array(122, 65), Array(122, 63), Array(128, 57), Array(144, 47), Array(154, 41), Array(165, 34), Array(171, 28), Array(173, 25) );
        $this->dtw->setCheck($check);
        $minCost = $this->dtw->getMinCost();
        $this->assertTrue($minCost > 0);
    }

    public function testLetterCheckO() {
        // o
        $reference = Array( Array(126, 95), Array(125, 95), Array(124, 95), Array(123, 95), Array(122, 95), Array(120, 95), Array(119, 94), Array(117, 93), Array(115, 92), Array(113, 91), Array(107, 87), Array(105, 85), Array(103, 80), Array(102, 79), Array(102, 77), Array(101, 76), Array(101, 74), Array(101, 72), Array(101, 71), Array(101, 69), Array(101, 67), Array(102, 60), Array(105, 53), Array(115, 48), Array(117, 48), Array(118, 47), Array(131, 46), Array(138, 46), Array(142, 47), Array(143, 47), Array(144, 48), Array(146, 49), Array(147, 50), Array(148, 52), Array(149, 53), Array(150, 54), Array(150, 55), Array(151, 61), Array(151, 63), Array(151, 73), Array(151, 81), Array(151, 88), Array(149, 90), Array(148, 91), Array(147, 94), Array(146, 95), Array(145, 96), Array(144, 96), Array(142, 97), Array(138, 98), Array(132, 98), Array(127, 98), Array(125, 98), Array(123, 98), Array(122, 98), Array(120, 98) );
        $this->dtw->setReference($reference);
        // a
        $check = Array( Array(129, 68), Array(127, 66), Array(126, 65), Array(125, 65), Array(123, 64), Array(122, 63), Array(121, 63), Array(118, 63), Array(117, 63), Array(115, 63), Array(113, 63), Array(111, 63), Array(110, 63), Array(109, 63), Array(108, 63), Array(104, 64), Array(101, 65), Array(95, 69), Array(92, 77), Array(92, 82), Array(94, 86), Array(97, 90), Array(100, 91), Array(103, 91), Array(105, 91), Array(116, 82), Array(120, 78), Array(126, 69), Array(130, 63), Array(131, 56), Array(131, 55), Array(132, 54), Array(132, 53), Array(132, 54), Array(132, 56), Array(132, 58), Array(132, 60), Array(132, 65), Array(132, 66), Array(132, 69), Array(132, 71), Array(132, 79), Array(132, 81), Array(132, 90), Array(132, 92), Array(134, 94), Array(135, 95), Array(137, 95) );
        $this->dtw->setCheck($check);
        $minCost_a = $this->dtw->getMinCost();
        // c
        $check = Array( Array(128, 42), Array(127, 42), Array(126, 42), Array(125, 42), Array(125, 41), Array(123, 40), Array(122, 38), Array(121, 38), Array(119, 37), Array(118, 37), Array(116, 36), Array(115, 36), Array(112, 35), Array(110, 35), Array(107, 35), Array(99, 35), Array(91, 35), Array(81, 36), Array(71, 41), Array(66, 47), Array(64, 54), Array(63, 55), Array(66, 65), Array(66, 67), Array(69, 70), Array(70, 71), Array(72, 72), Array(77, 75), Array(80, 77), Array(89, 81), Array(95, 82), Array(100, 82), Array(108, 82), Array(112, 82), Array(113, 82), Array(117, 82) );
        $this->dtw->setCheck($check);
        $minCost_c = $this->dtw->getMinCost();
        // попытка о
        $check = Array( Array(129, 73), Array(127, 73), Array(126, 73), Array(125, 73), Array(123, 73), Array(121, 73), Array(120, 73), Array(115, 72), Array(114, 70), Array(105, 65), Array(104, 63), Array(103, 62), Array(102, 54), Array(102, 50), Array(102, 45), Array(103, 41), Array(114, 35), Array(117, 34), Array(118, 33), Array(122, 32), Array(126, 31), Array(129, 31), Array(136, 32), Array(140, 34), Array(144, 38), Array(145, 39), Array(147, 42), Array(152, 48), Array(154, 54), Array(154, 64), Array(154, 67), Array(154, 69), Array(153, 70), Array(151, 73), Array(146, 74), Array(142, 76), Array(139, 77), Array(133, 77), Array(131, 77), Array(128, 77), Array(126, 77), Array(124, 77), Array(122, 77) );
        $this->dtw->setCheck($check);
        $minCost_o = $this->dtw->getMinCost();
        // б
        $check = Array( Array(131, 79), Array(130, 79), Array(129, 79), Array(127, 79), Array(125, 79), Array(124, 79), Array(121, 79), Array(118, 80), Array(116, 80), Array(113, 80), Array(112, 80), Array(105, 82), Array(98, 88), Array(98, 94), Array(99, 99), Array(103, 101), Array(109, 102), Array(114, 102), Array(126, 99), Array(128, 98), Array(131, 95), Array(133, 90), Array(132, 81), Array(127, 75), Array(121, 67), Array(109, 58), Array(102, 52), Array(99, 48), Array(93, 43), Array(93, 40), Array(98, 38), Array(108, 35), Array(124, 34), Array(139, 33), Array(149, 33), Array(157, 32), Array(160, 32) );
        $this->dtw->setCheck($check);
        $minCost_b = $this->dtw->getMinCost();
        $this->assertTrue($minCost_o == min($minCost_a, $minCost_b, $minCost_c, $minCost_o));
    }

    public function testLetterCheckA() {
        // a
        $reference = Array( Array(147, 56), Array(147, 55), Array(146, 55), Array(144, 53), Array(143, 51), Array(141, 50), Array(140, 49), Array(137, 47), Array(131, 45), Array(123, 44), Array(122, 44), Array(120, 44), Array(118, 44), Array(117, 44), Array(113, 48), Array(112, 50), Array(108, 56), Array(108, 65), Array(108, 67), Array(113, 78), Array(116, 80), Array(121, 82), Array(128, 82), Array(136, 81), Array(140, 79), Array(142, 73), Array(145, 59), Array(146, 52), Array(146, 49), Array(146, 46), Array(146, 44), Array(146, 43), Array(146, 45), Array(146, 47), Array(146, 57), Array(146, 63), Array(147, 64), Array(149, 80), Array(153, 84), Array(158, 86), Array(159, 86), Array(160, 86), Array(163, 86), Array(165, 84) );
        $this->dtw->setReference($reference);
        // попытка a
        $check = Array( Array(152, 46), Array(152, 44), Array(152, 43), Array(152, 41), Array(152, 40), Array(152, 38), Array(152, 37), Array(152, 36), Array(151, 33), Array(149, 32), Array(147, 28), Array(142, 26), Array(141, 26), Array(137, 26), Array(134, 26), Array(133, 26), Array(129, 26), Array(128, 26), Array(121, 32), Array(119, 34), Array(119, 35), Array(119, 38), Array(118, 39), Array(118, 42), Array(117, 50), Array(117, 53), Array(120, 58), Array(122, 61), Array(132, 69), Array(138, 70), Array(139, 70), Array(143, 70), Array(152, 58), Array(153, 56), Array(153, 55), Array(154, 54), Array(156, 48), Array(157, 37), Array(157, 28), Array(157, 26), Array(156, 26), Array(156, 25), Array(156, 26), Array(155, 28), Array(155, 30), Array(154, 32), Array(154, 34), Array(154, 40), Array(154, 45), Array(155, 51), Array(155, 56), Array(156, 59), Array(157, 61), Array(157, 63), Array(157, 64), Array(158, 67) );
        $this->dtw->setCheck($check);
        $minCost_a = $this->dtw->getMinCost();
        // c
        $check = Array( Array(162, 52), Array(162, 51), Array(161, 51), Array(160, 49), Array(154, 47), Array(146, 44), Array(140, 41), Array(139, 41), Array(135, 41), Array(134, 41), Array(130, 41), Array(129, 41), Array(124, 43), Array(123, 44), Array(116, 52), Array(112, 59), Array(110, 70), Array(111, 78), Array(112, 80), Array(115, 83), Array(117, 84), Array(118, 85), Array(121, 87), Array(123, 87), Array(124, 88), Array(126, 89), Array(127, 89), Array(129, 89), Array(133, 90), Array(141, 92), Array(149, 93), Array(155, 93), Array(159, 91), Array(161, 91) );
        $this->dtw->setCheck($check);
        $minCost_c = $this->dtw->getMinCost();
        // о
        $check = Array( Array(132, 38), Array(131, 37), Array(130, 36), Array(129, 36), Array(129, 35), Array(128, 35), Array(127, 35), Array(126, 35), Array(125, 35), Array(124, 35), Array(123, 36), Array(118, 37), Array(112, 40), Array(110, 42), Array(110, 46), Array(108, 51), Array(108, 58), Array(108, 59), Array(108, 60), Array(110, 67), Array(130, 80), Array(133, 80), Array(134, 80), Array(137, 80), Array(139, 80), Array(140, 80), Array(142, 80), Array(145, 79), Array(147, 78), Array(149, 78), Array(150, 77), Array(151, 76), Array(151, 75), Array(153, 74), Array(155, 72), Array(155, 71), Array(157, 66), Array(159, 57), Array(159, 55), Array(158, 51), Array(158, 49), Array(156, 49), Array(155, 47), Array(152, 44), Array(146, 41), Array(141, 38), Array(136, 35), Array(129, 34), Array(125, 33), Array(123, 32), Array(121, 32), Array(120, 32) );
        $this->dtw->setCheck($check);
        $minCost_o = $this->dtw->getMinCost();
        // б
        $check = Array( Array(164, 77), Array(161, 77), Array(160, 77), Array(156, 77), Array(154, 77), Array(151, 77), Array(146, 77), Array(145, 77), Array(141, 78), Array(139, 80), Array(138, 81), Array(137, 84), Array(138, 89), Array(143, 92), Array(151, 94), Array(158, 98), Array(174, 99), Array(177, 99), Array(181, 91), Array(182, 89), Array(182, 83), Array(176, 78), Array(173, 74), Array(169, 72), Array(167, 71), Array(160, 66), Array(152, 62), Array(143, 56), Array(137, 51), Array(137, 47), Array(137, 42), Array(142, 36), Array(153, 30), Array(163, 27), Array(166, 27), Array(168, 27), Array(171, 27), Array(173, 28), Array(174, 28) );
        $this->dtw->setCheck($check);
        $minCost_b = $this->dtw->getMinCost();
        $this->assertTrue($minCost_a == min($minCost_a, $minCost_b, $minCost_c, $minCost_o));
    }

    public function testLetterCheck_test() {
        $reference = Array( Array(63, 48), Array(63, 51), Array(63, 56), Array(63, 68), Array(64, 74), Array(64, 84), Array(64, 87), Array(64, 89), Array(64, 90), Array(50, 48), Array(51, 48), Array(53, 48), Array(57, 46), Array(63, 45), Array(78, 44), Array(81, 44), Array(84, 44), Array(89, 74), Array(91, 74), Array(94, 74), Array(95, 74), Array(97, 73), Array(98, 73), Array(99, 72), Array(100, 72), Array(102, 70), Array(102, 69), Array(102, 68), Array(99, 65), Array(98, 63), Array(95, 62), Array(93, 62), Array(92, 62), Array(90, 62), Array(88, 65), Array(88, 67), Array(88, 68), Array(88, 71), Array(90, 79), Array(91, 79), Array(92, 80), Array(97, 83), Array(98, 83), Array(99, 83), Array(102, 83), Array(103, 83), Array(105, 82), Array(108, 81), Array(131, 61), Array(130, 61), Array(128, 61), Array(127, 61), Array(126, 61), Array(125, 61), Array(121, 63), Array(120, 67), Array(120, 70), Array(121, 72), Array(122, 73), Array(125, 76), Array(133, 79), Array(135, 79), Array(138, 80), Array(139, 80), Array(139, 79), Array(152, 58), Array(152, 60), Array(152, 66), Array(152, 69), Array(152, 72), Array(152, 75), Array(152, 78), Array(152, 82), Array(152, 83), Array(143, 62), Array(150, 60), Array(167, 58), Array(174, 57), Array(176, 57), Array(178, 56), Array(179, 56) );
        $this->dtw->setReference($reference);
        // попытка прописными
        $check = Array( Array(44, 43), Array(44, 44), Array(44, 45), Array(44, 46), Array(44, 48), Array(44, 50), Array(44, 51), Array(45, 58), Array(45, 65), Array(45, 63), Array(45, 61), Array(45, 58), Array(45, 57), Array(53, 47), Array(54, 47), Array(56, 47), Array(57, 47), Array(58, 47), Array(59, 47), Array(61, 47), Array(63, 48), Array(64, 49), Array(66, 52), Array(66, 53), Array(66, 57), Array(66, 56), Array(67, 55), Array(68, 53), Array(69, 51), Array(70, 50), Array(74, 48), Array(77, 48), Array(78, 48), Array(79, 48), Array(81, 50), Array(81, 57), Array(82, 63), Array(82, 64), Array(91, 59), Array(93, 59), Array(93, 58), Array(94, 58), Array(95, 58), Array(97, 55), Array(98, 52), Array(98, 51), Array(97, 51), Array(97, 50), Array(96, 50), Array(95, 50), Array(94, 50), Array(91, 51), Array(89, 53), Array(89, 55), Array(89, 57), Array(89, 58), Array(90, 60), Array(93, 61), Array(98, 62), Array(99, 62), Array(108, 60), Array(109, 60), Array(111, 59), Array(115, 57), Array(116, 56), Array(116, 55), Array(118, 54), Array(118, 53), Array(121, 48), Array(122, 48), Array(122, 47), Array(122, 46), Array(123, 46), Array(124, 46), Array(125, 46), Array(123, 46), Array(121, 46), Array(120, 47), Array(119, 47), Array(118, 47), Array(118, 48), Array(118, 49), Array(118, 51), Array(118, 52), Array(118, 53), Array(119, 59), Array(119, 60), Array(120, 60), Array(124, 61), Array(127, 61), Array(131, 61), Array(132, 61), Array(137, 59), Array(140, 58), Array(143, 53), Array(144, 51), Array(145, 50), Array(147, 47), Array(147, 46), Array(148, 44), Array(149, 42), Array(151, 41), Array(151, 42), Array(151, 44), Array(151, 46), Array(151, 47), Array(151, 49), Array(151, 50), Array(151, 52), Array(151, 54), Array(150, 59), Array(148, 59), Array(146, 59), Array(145, 57), Array(143, 54), Array(141, 52), Array(140, 51), Array(138, 48), Array(138, 46), Array(135, 40), Array(134, 40), Array(133, 37), Array(132, 37), Array(132, 36), Array(134, 36), Array(137, 36), Array(151, 36), Array(154, 36), Array(165, 36), Array(168, 36), Array(173, 36) );
        $this->dtw->setCheck($check);
        $minCost_1 = $this->dtw->getMinCost();
        // попытка чуть более правильная
        $check = Array( Array(51, 34), Array(51, 35), Array(51, 36), Array(52, 38), Array(52, 41), Array(53, 43), Array(53, 46), Array(54, 49), Array(54, 53), Array(55, 58), Array(57, 71), Array(58, 80), Array(28, 37), Array(29, 37), Array(32, 37), Array(37, 37), Array(43, 37), Array(48, 37), Array(71, 36), Array(85, 35), Array(86, 35), Array(83, 74), Array(84, 74), Array(85, 74), Array(87, 73), Array(88, 72), Array(90, 72), Array(92, 71), Array(93, 70), Array(96, 70), Array(101, 66), Array(105, 63), Array(107, 60), Array(107, 59), Array(106, 59), Array(105, 59), Array(102, 59), Array(96, 59), Array(95, 60), Array(94, 61), Array(94, 63), Array(94, 64), Array(94, 65), Array(94, 67), Array(95, 68), Array(95, 69), Array(95, 70), Array(96, 70), Array(99, 74), Array(108, 78), Array(110, 78), Array(111, 78), Array(115, 78), Array(119, 77), Array(120, 76), Array(121, 75), Array(149, 63), Array(148, 63), Array(146, 62), Array(145, 62), Array(144, 62), Array(141, 63), Array(140, 63), Array(139, 64), Array(137, 64), Array(136, 65), Array(135, 65), Array(134, 65), Array(133, 66), Array(133, 67), Array(132, 72), Array(132, 73), Array(132, 75), Array(132, 76), Array(132, 78), Array(133, 79), Array(135, 79), Array(140, 80), Array(144, 81), Array(151, 81), Array(155, 79), Array(156, 79), Array(157, 79), Array(172, 60), Array(172, 61), Array(172, 63), Array(172, 65), Array(171, 67), Array(170, 69), Array(170, 71), Array(170, 73), Array(170, 77), Array(170, 78), Array(170, 80), Array(169, 80), Array(168, 80), Array(167, 80), Array(155, 60), Array(156, 60), Array(160, 60), Array(163, 59), Array(168, 59), Array(173, 59), Array(179, 59), Array(193, 59), Array(201, 59), Array(202, 59) );
        $this->dtw->setCheck($check);
        $minCost_2 = $this->dtw->getMinCost();
        // наиболее похожая попытка
        $check = Array( Array(63, 35), Array(63, 38), Array(63, 45), Array(65, 54), Array(65, 58), Array(65, 63), Array(68, 77), Array(68, 78), Array(45, 37), Array(45, 36), Array(46, 36), Array(49, 35), Array(53, 34), Array(56, 34), Array(73, 31), Array(78, 31), Array(84, 31), Array(85, 31), Array(90, 66), Array(91, 66), Array(92, 66), Array(93, 65), Array(95, 64), Array(98, 63), Array(101, 61), Array(101, 60), Array(101, 59), Array(98, 57), Array(97, 57), Array(92, 55), Array(90, 55), Array(89, 55), Array(88, 55), Array(86, 57), Array(84, 64), Array(84, 65), Array(85, 66), Array(93, 71), Array(100, 74), Array(103, 75), Array(105, 75), Array(109, 74), Array(109, 73), Array(109, 72), Array(137, 55), Array(136, 55), Array(136, 54), Array(135, 54), Array(134, 54), Array(133, 54), Array(132, 54), Array(131, 54), Array(126, 54), Array(122, 55), Array(118, 60), Array(116, 60), Array(116, 63), Array(117, 67), Array(123, 71), Array(125, 73), Array(130, 75), Array(131, 76), Array(136, 78), Array(137, 78), Array(138, 79), Array(139, 79), Array(158, 54), Array(158, 55), Array(158, 56), Array(158, 57), Array(158, 58), Array(158, 65), Array(158, 77), Array(158, 82), Array(158, 83), Array(149, 52), Array(150, 52), Array(152, 51), Array(153, 51), Array(155, 50), Array(157, 50), Array(161, 50), Array(164, 50), Array(168, 50), Array(181, 49), Array(183, 49), Array(184, 49) );
        $this->dtw->setCheck($check);
        $minCost_3 = $this->dtw->getMinCost();
        $this->assertTrue($minCost_3 == min($minCost_1, $minCost_2, $minCost_3));
    }

    public function testTestTestCheck_test() {
        $reference = Array( Array(108, 43), Array(108, 49), Array(108, 62), Array(108, 68), Array(109, 81), Array(88, 53), Array(88, 52), Array(89, 52), Array(92, 51), Array(95, 49), Array(98, 48), Array(101, 47), Array(106, 45), Array(109, 43), Array(126, 38), Array(127, 37), Array(126, 61), Array(127, 61), Array(128, 61), Array(131, 60), Array(133, 59), Array(136, 57), Array(139, 57), Array(140, 56), Array(142, 55), Array(145, 48), Array(145, 47), Array(135, 48), Array(134, 49), Array(131, 57), Array(131, 58), Array(131, 60), Array(133, 66), Array(134, 66), Array(136, 67), Array(143, 68), Array(144, 68), Array(144, 67), Array(144, 66), Array(157, 52), Array(156, 52), Array(155, 54), Array(146, 61), Array(145, 62), Array(154, 72), Array(155, 73), Array(167, 73), Array(167, 72), Array(177, 55), Array(177, 57), Array(177, 60), Array(177, 62), Array(177, 66), Array(177, 74), Array(177, 79), Array(167, 58), Array(167, 57), Array(169, 56), Array(172, 56), Array(203, 56)  );
        $this->dtw->setReference($reference);
        // попытка 1
        $check = Array( Array(56, 58), Array(56, 60), Array(56, 63), Array(56, 67), Array(56, 70), Array(56, 72), Array(56, 77), Array(56, 79), Array(57, 81), Array(57, 82), Array(57, 84), Array(57, 82), Array(57, 79), Array(57, 67), Array(58, 65), Array(59, 61), Array(65, 57), Array(68, 57), Array(71, 59), Array(81, 72), Array(81, 81), Array(81, 82), Array(81, 75), Array(84, 70), Array(85, 66), Array(87, 64), Array(88, 62), Array(90, 60), Array(92, 60), Array(95, 59), Array(100, 59), Array(110, 72), Array(110, 80), Array(110, 81), Array(130, 78), Array(132, 78), Array(133, 78), Array(136, 78), Array(137, 77), Array(138, 75), Array(144, 69), Array(145, 67), Array(145, 64), Array(134, 62), Array(132, 63), Array(127, 68), Array(127, 71), Array(127, 72), Array(127, 73), Array(128, 75), Array(129, 76), Array(131, 77), Array(133, 78), Array(134, 78), Array(142, 80), Array(145, 80), Array(146, 80), Array(149, 80), Array(150, 80), Array(150, 79), Array(152, 78), Array(172, 65), Array(171, 65), Array(170, 65), Array(169, 65), Array(166, 65), Array(164, 66), Array(163, 67), Array(163, 69), Array(162, 74), Array(162, 75), Array(164, 78), Array(165, 78), Array(174, 83), Array(176, 83), Array(178, 84), Array(184, 83), Array(191, 61), Array(191, 62), Array(191, 65), Array(191, 69), Array(191, 73), Array(191, 77), Array(191, 81), Array(191, 84), Array(191, 90), Array(191, 92), Array(192, 93), Array(192, 94), Array(190, 90), Array(181, 65), Array(182, 65), Array(184, 65), Array(186, 64), Array(189, 63), Array(191, 63), Array(195, 63), Array(197, 63), Array(199, 63), Array(203, 63) );
        $this->dtw->setCheck($check);
        $minCost_1 = $this->dtw->getMinCost();
        // попытка 2(+)
        $check = Array( Array(56, 48), Array(56, 49), Array(56, 52), Array(56, 55), Array(56, 59), Array(56, 62), Array(58, 87), Array(33, 51), Array(35, 50), Array(38, 49), Array(39, 49), Array(40, 48), Array(43, 47), Array(53, 44), Array(56, 42), Array(70, 40), Array(73, 39), Array(74, 71), Array(75, 71), Array(76, 71), Array(83, 69), Array(85, 69), Array(87, 69), Array(89, 69), Array(91, 69), Array(95, 67), Array(96, 67), Array(96, 66), Array(97, 65), Array(97, 64), Array(97, 62), Array(96, 62), Array(95, 62), Array(93, 61), Array(92, 61), Array(90, 61), Array(89, 61), Array(88, 61), Array(86, 61), Array(84, 62), Array(82, 67), Array(82, 69), Array(82, 71), Array(82, 72), Array(83, 73), Array(84, 74), Array(85, 74), Array(90, 79), Array(92, 80), Array(99, 81), Array(100, 81), Array(101, 81), Array(102, 81), Array(127, 59), Array(126, 59), Array(125, 59), Array(124, 59), Array(123, 59), Array(122, 59), Array(121, 59), Array(120, 61), Array(119, 62), Array(119, 63), Array(115, 70), Array(115, 71), Array(115, 72), Array(115, 73), Array(117, 76), Array(124, 79), Array(125, 79), Array(126, 79), Array(130, 79), Array(131, 78), Array(158, 61), Array(158, 63), Array(158, 65), Array(158, 67), Array(158, 69), Array(158, 71), Array(158, 74), Array(158, 77), Array(158, 82), Array(158, 86), Array(142, 64), Array(143, 64), Array(144, 63), Array(160, 62), Array(165, 62), Array(169, 61), Array(172, 61), Array(175, 61), Array(178, 60) );
        $this->dtw->setCheck($check);
        $minCost_2 = $this->dtw->getMinCost();
        // Попытка 3
        $check = Array( Array(44, 37), Array(44, 41), Array(43, 47), Array(43, 50), Array(43, 53), Array(43, 56), Array(43, 78), Array(19, 20), Array(19, 21), Array(21, 21), Array(22, 22), Array(24, 23), Array(25, 23), Array(26, 25), Array(31, 28), Array(52, 41), Array(53, 42), Array(55, 43), Array(57, 43), Array(59, 44), Array(60, 44), Array(61, 45), Array(94, 59), Array(93, 59), Array(92, 59), Array(91, 59), Array(90, 59), Array(89, 59), Array(88, 59), Array(86, 59), Array(81, 59), Array(80, 59), Array(79, 59), Array(76, 60), Array(75, 61), Array(75, 62), Array(80, 64), Array(83, 64), Array(88, 64), Array(87, 64), Array(86, 64), Array(85, 64), Array(84, 64), Array(83, 64), Array(82, 64), Array(81, 64), Array(81, 65), Array(80, 65), Array(80, 66), Array(80, 67), Array(80, 70), Array(80, 71), Array(80, 72), Array(81, 73), Array(82, 73), Array(83, 73), Array(93, 74), Array(94, 74), Array(95, 74), Array(133, 60), Array(132, 60), Array(131, 60), Array(129, 60), Array(126, 60), Array(123, 60), Array(120, 60), Array(112, 60), Array(111, 60), Array(110, 60), Array(110, 61), Array(110, 62), Array(110, 64), Array(110, 66), Array(110, 74), Array(110, 78), Array(110, 80), Array(110, 83), Array(110, 84), Array(111, 84), Array(112, 84), Array(113, 84), Array(114, 84), Array(115, 84), Array(117, 84), Array(121, 83), Array(122, 83), Array(124, 82), Array(125, 82), Array(126, 82), Array(127, 82), Array(128, 82), Array(143, 66), Array(143, 67), Array(143, 68), Array(143, 69), Array(143, 70), Array(143, 71), Array(143, 72), Array(143, 75), Array(143, 77), Array(143, 78), Array(144, 79), Array(144, 73), Array(144, 70), Array(144, 68), Array(144, 65), Array(144, 62), Array(144, 61), Array(144, 59), Array(144, 58), Array(145, 58), Array(147, 57), Array(156, 60), Array(157, 61), Array(159, 62), Array(161, 66), Array(161, 73), Array(161, 70), Array(162, 68), Array(163, 66), Array(163, 65), Array(165, 64), Array(166, 64), Array(167, 64), Array(171, 64), Array(173, 64), Array(174, 64), Array(175, 65), Array(176, 66), Array(178, 70), Array(178, 75) );
        $this->dtw->setCheck($check);
        $minCost_3 = $this->dtw->getMinCost();
        // Попытка 4
        $check = Array( Array(46, 50), Array(46, 51), Array(46, 53), Array(46, 59), Array(46, 63), Array(46, 66), Array(46, 70), Array(46, 73), Array(46, 76), Array(46, 83), Array(46, 85), Array(46, 87), Array(46, 89), Array(46, 90), Array(46, 92), Array(46, 96), Array(49, 86), Array(51, 80), Array(53, 75), Array(57, 66), Array(59, 63), Array(61, 61), Array(66, 56), Array(67, 56), Array(68, 56), Array(69, 56), Array(76, 58), Array(77, 58), Array(85, 70), Array(85, 71), Array(85, 74), Array(85, 87), Array(85, 85), Array(85, 84), Array(85, 82), Array(85, 78), Array(86, 76), Array(86, 74), Array(91, 68), Array(92, 67), Array(93, 66), Array(93, 64), Array(94, 63), Array(95, 62), Array(98, 61), Array(102, 61), Array(103, 61), Array(104, 63), Array(105, 63), Array(106, 64), Array(107, 65), Array(108, 71), Array(108, 72), Array(108, 75), Array(107, 87), Array(109, 87), Array(110, 87), Array(112, 87), Array(119, 85), Array(120, 85), Array(121, 84), Array(122, 84), Array(123, 83), Array(131, 80), Array(132, 78), Array(139, 72), Array(139, 71), Array(140, 70), Array(136, 66), Array(134, 66), Array(131, 66), Array(129, 70), Array(131, 82), Array(132, 84), Array(133, 84), Array(133, 85), Array(134, 85), Array(135, 85), Array(142, 86), Array(149, 86), Array(150, 86), Array(151, 86), Array(152, 86), Array(154, 83), Array(154, 82), Array(155, 81), Array(161, 71), Array(164, 65), Array(165, 65), Array(166, 65), Array(165, 65), Array(164, 65), Array(163, 65), Array(162, 65), Array(161, 65), Array(154, 68), Array(154, 69), Array(152, 71), Array(154, 77), Array(156, 78), Array(163, 79), Array(166, 79), Array(168, 79), Array(171, 79), Array(180, 77), Array(182, 76), Array(183, 74), Array(185, 74), Array(185, 72), Array(187, 69), Array(189, 66), Array(189, 65), Array(190, 59), Array(190, 58), Array(189, 79), Array(189, 80), Array(189, 82), Array(189, 87), Array(189, 88), Array(189, 89), Array(189, 90), Array(185, 85), Array(182, 80), Array(181, 77), Array(179, 75), Array(177, 72), Array(175, 68), Array(174, 66), Array(173, 62), Array(174, 62), Array(175, 61), Array(198, 59), Array(206, 59), Array(211, 59), Array(214, 59), Array(215, 59), Array(218, 59) );
        $this->dtw->setCheck($check);
        $minCost_4 = $this->dtw->getMinCost();
        $this->assertTrue($minCost_1 == min($minCost_1, $minCost_2, $minCost_3, $minCost_4));
    }

    public function testLetter_test() {
        $reference = Array( Array(113, 111), Array(113, 110), Array(116, 106), Array(118, 103), Array(120, 101), Array(123, 99), Array(124, 96), Array(126, 94), Array(131, 86), Array(135, 82), Array(135, 78), Array(137, 75), Array(139, 72), Array(140, 69), Array(147, 55), Array(149, 52), Array(153, 43), Array(153, 44), Array(153, 45), Array(153, 48), Array(154, 49), Array(154, 51), Array(154, 64), Array(154, 66), Array(155, 76), Array(157, 94), Array(157, 103), Array(158, 103), Array(158, 105), Array(159, 105), Array(159, 106), Array(123, 87), Array(124, 87), Array(125, 87), Array(128, 87), Array(132, 87), Array(135, 87), Array(140, 87), Array(143, 87), Array(144, 87), Array(145, 87), Array(147, 86), Array(151, 86), Array(156, 86), Array(157, 86), Array(160, 85) );
        $this->dtw->setReference($reference);
        // попытка сравнения со сдвинутым оригиналом
        $check = Array();
        $move = 10;
        foreach($reference as $item) {
            $checkTmp = Array($item[0] + $move, $item[1] + $move);
            $new_check[] = $checkTmp;
        }
        $this->dtw->setCheck($new_check);
        $min_cost_move = $this->dtw->getMinCost();
        // попытка 1
        $check = Array( Array(114, 110), Array(114, 108), Array(114, 107), Array(115, 106), Array(116, 104), Array(117, 102), Array(118, 101), Array(120, 97), Array(121, 94), Array(122, 91), Array(123, 88), Array(125, 84), Array(127, 81), Array(131, 75), Array(139, 61), Array(140, 55), Array(141, 53), Array(144, 45), Array(144, 43), Array(145, 42), Array(146, 40), Array(147, 38), Array(147, 37), Array(147, 34), Array(151, 24), Array(151, 25), Array(152, 27), Array(152, 29), Array(152, 31), Array(153, 34), Array(153, 63), Array(153, 70), Array(153, 88), Array(153, 89), Array(154, 90), Array(154, 91), Array(154, 92), Array(154, 93), Array(154, 95), Array(154, 96), Array(121, 77), Array(122, 77), Array(123, 77), Array(124, 76), Array(125, 76), Array(126, 76), Array(129, 76), Array(130, 76), Array(131, 76), Array(132, 76), Array(133, 76), Array(135, 76), Array(139, 76), Array(140, 76), Array(148, 75), Array(150, 75), Array(152, 75), Array(154, 75), Array(155, 75), Array(157, 75), Array(158, 75), Array(159, 75) );
        $this->dtw->setCheck($check);
        $minCost_1 = $this->dtw->getMinCost();
        // попытка 1' перенос точки
        $new_check = Array();
        $move = -10;
        foreach($check as $item) {
            $checkTmp = Array($item[0] + $move, $item[1] + $move);
            $new_check[] = $checkTmp;
        }
        $this->dtw->setCheck($new_check);
        $min_cost_move = $this->dtw->getMinCost();
        // попытка 1' перенос точки 2
        $new_check = Array();
        $move = 30;
        foreach($check as $item) {
            $checkTmp = Array($item[0] + $move, $item[1] + $move);
            $new_check[] = $checkTmp;
        }
        $this->dtw->setCheck($new_check);
        $min_cost_move = $this->dtw->getMinCost();
        // попытка 2(+)
        $check = Array( Array(163, 75), Array(163, 74), Array(163, 73), Array(163, 72), Array(162, 72), Array(160, 70), Array(158, 68), Array(157, 68), Array(155, 67), Array(153, 66), Array(152, 66), Array(150, 66), Array(149, 66), Array(148, 66), Array(146, 66), Array(145, 67), Array(143, 68), Array(142, 68), Array(141, 68), Array(139, 70), Array(136, 81), Array(135, 84), Array(134, 86), Array(134, 89), Array(134, 90), Array(134, 93), Array(137, 94), Array(139, 95), Array(146, 95), Array(150, 95), Array(161, 94), Array(162, 94), Array(165, 92), Array(170, 83), Array(171, 74), Array(171, 68), Array(171, 61), Array(170, 53), Array(169, 52), Array(167, 50), Array(151, 41), Array(141, 41), Array(139, 41), Array(138, 41), Array(137, 41), Array(134, 41), Array(170, 77), Array(170, 78), Array(170, 79), Array(170, 80), Array(170, 81), Array(170, 84), Array(170, 86), Array(170, 87), Array(171, 90), Array(175, 94), Array(178, 97), Array(185, 99), Array(187, 99) );
        $this->dtw->setCheck($check);
        $minCost_2 = $this->dtw->getMinCost();
        // Попытка 3
        $check = Array( Array(127, 110), Array(127, 109), Array(127, 108), Array(127, 106), Array(127, 102), Array(127, 99), Array(127, 90), Array(127, 85), Array(127, 81), Array(127, 70), Array(127, 63), Array(127, 40), Array(127, 37), Array(127, 35), Array(130, 69), Array(131, 69), Array(132, 69), Array(132, 68), Array(134, 67), Array(135, 67), Array(136, 67), Array(138, 66), Array(139, 65), Array(148, 63), Array(150, 63), Array(155, 63), Array(168, 63), Array(175, 64), Array(176, 64), Array(181, 72), Array(181, 77), Array(172, 88), Array(170, 88), Array(169, 89), Array(166, 90), Array(158, 93), Array(155, 94), Array(154, 94), Array(153, 95), Array(152, 95), Array(151, 95), Array(149, 96), Array(145, 96), Array(139, 96), Array(138, 96), Array(137, 96), Array(134, 96), Array(132, 96), Array(131, 96), Array(130, 96), Array(127, 37), Array(128, 37), Array(129, 36), Array(131, 36), Array(133, 36), Array(135, 35), Array(137, 35), Array(140, 35), Array(142, 35), Array(146, 35), Array(148, 35), Array(149, 35), Array(151, 35), Array(157, 34), Array(168, 34) );
        $this->dtw->setCheck($check);
        $minCost_3 = $this->dtw->getMinCost();
        // Попытка 4
        $check = Array( Array(148, 81), Array(148, 80), Array(148, 79), Array(148, 78), Array(147, 77), Array(147, 71), Array(147, 68), Array(154, 64), Array(158, 64), Array(163, 65), Array(164, 66), Array(165, 67), Array(168, 85), Array(168, 89), Array(163, 94), Array(152, 100), Array(151, 100), Array(150, 100), Array(137, 95), Array(136, 93), Array(134, 73), Array(151, 46), Array(172, 46), Array(174, 47), Array(177, 49), Array(177, 51), Array(181, 57), Array(182, 61), Array(183, 70), Array(183, 76), Array(183, 88), Array(183, 89), Array(177, 102), Array(169, 106), Array(160, 110), Array(157, 111), Array(152, 112), Array(151, 112), Array(145, 113), Array(141, 112), Array(136, 108), Array(132, 105), Array(126, 97), Array(126, 94), Array(125, 92), Array(125, 89), Array(125, 86), Array(125, 77), Array(125, 75), Array(125, 68), Array(129, 54), Array(142, 37), Array(143, 35), Array(145, 33), Array(153, 27), Array(155, 26), Array(169, 22), Array(170, 22), Array(174, 22), Array(188, 24), Array(190, 25), Array(198, 29), Array(206, 35), Array(211, 41), Array(212, 58), Array(212, 63), Array(212, 68), Array(212, 70), Array(212, 75), Array(207, 86), Array(206, 88), Array(202, 96), Array(190, 109), Array(188, 111), Array(174, 121), Array(168, 123), Array(160, 126), Array(159, 127), Array(158, 127), Array(153, 129), Array(151, 129), Array(150, 129), Array(148, 129), Array(145, 129), Array(138, 127), Array(136, 126), Array(115, 111), Array(111, 108), Array(106, 90), Array(106, 87), Array(106, 79), Array(117, 52), Array(120, 48), Array(131, 28), Array(132, 27), Array(137, 23), Array(139, 22), Array(143, 20) );
        $this->dtw->setCheck($check);
        $minCost_4 = $this->dtw->getMinCost();
        $this->assertTrue($minCost_1 == min($minCost_1, $minCost_2, $minCost_3, $minCost_4));
        // Попытка 5 data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAASwAAACWCAYAAABkW7XSAAACwElEQVR4nO3dQU7cMACG0f/m7YEQR6h6A25Qdceu6gYhxAzugkYClDDJJI6d9j0pEqCZwWw+mYzHTgAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAABY7ZSktB4EwFwlybn1IADmOEewgIMQLOAwBAs4jHOSl9aDAJhDsIDDGILl30Kge2ZYwGGYXQGHcIpgAQcxrHIXLKB7ggUcwjmCBRxEyes9LMECujbMroavBQvoVkny/PdrwQK69ZL3m/YJFtCtkuThzfeCBXTp4+wqESygUyXJ3YefCRbQnbHZVSJYQIemTseZChlAEyXTW8gIFtCNu3weJMECuvHZ7Cp5v+odoJmHXI6R05+BLlyaXSXJYwQLaOw580J0P/NxANUMe11dchvBAhpael9KsIBmSpKnhY+/rTQWgElzbrSPPee+wlgAJl27pqrk9d1CgN1cey9q2N8dYBdz3xXc+rkAi6z9POA1970AFnvK+mUJggXsYukSho+G3RwEC1hsyb2kLUJT3rzOryQ/knxL8mXl6wL/gZL3EfnsmvOY4TpPXMPpzyXJ7yQ/k3xP8rXqXwn8U4aI1L7exgxgc1svQxAsoIoaWxoLFrC5LZYwjBEsYHNrlzBMESxgUzXXSgkWsJnaJ9sIFrCZ2juCChawiT12UhAsYLW9TmUWLGCVWksYxggWsEqtJQxjBAu42t7bvQgWcJXaSximfqdgAYu1ONRUsIDFWh0GIVjAInstYRgjWMBsey5hGCNYwGytDzIVLGCWFu8Kjo1BsICLSpLHxmMQLOCiljfaARYpSW5aDwLgErMr4BBuIlbAQez94WaAqzzG7Ao4iFafFwRY5BSzKwAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAoJ4/blIle/Oo0lkAAAAASUVORK5CYII=
        $check = Array( Array(148, 35), Array(148, 37), Array(147, 38), Array(146, 40), Array(145, 43), Array(136, 60), Array(116, 88), Array(110, 99), Array(147, 30), Array(147, 31), Array(147, 33), Array(147, 35), Array(147, 39), Array(147, 44), Array(147, 48), Array(147, 51), Array(148, 68), Array(148, 77), Array(148, 80), Array(148, 83), Array(148, 91), Array(119, 70), Array(121, 70), Array(123, 70), Array(125, 70), Array(128, 70), Array(130, 70), Array(133, 70), Array(140, 70), Array(146, 70), Array(147, 70), Array(147, 69), Array(148, 69), Array(149, 69), Array(150, 69), Array(151, 69), Array(152, 69), Array(153, 69) );
        $this->dtw->setCheck($check);
        $minCost_5 = $this->dtw->getMinCost();
        $this->assertTrue($minCost_1 == min($minCost_1, $minCost_2, $minCost_3, $minCost_4, $minCost_5));
    }

    public function testWord_test() {
        // оригинал data:image/png;base64,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
        $reference = Array( Array(43, 56), Array(44, 56), Array(44, 55), Array(44, 54), Array(44, 52), Array(43, 51), Array(42, 50), Array(41, 50), Array(40, 50), Array(39, 50), Array(31, 54), Array(30, 55), Array(30, 56), Array(29, 57), Array(27, 60), Array(27, 65), Array(27, 66), Array(27, 67), Array(27, 69), Array(26, 79), Array(26, 82), Array(30, 85), Array(33, 86), Array(38, 87), Array(42, 85), Array(42, 84), Array(42, 83), Array(45, 76), Array(51, 84), Array(52, 84), Array(53, 84), Array(54, 84), Array(56, 84), Array(57, 83), Array(59, 82), Array(65, 73), Array(66, 59), Array(66, 58), Array(66, 52), Array(67, 53), Array(67, 54), Array(68, 56), Array(68, 66), Array(70, 69), Array(70, 71), Array(74, 78), Array(76, 79), Array(78, 78), Array(81, 76), Array(86, 67), Array(87, 56), Array(87, 55), Array(87, 60), Array(92, 72), Array(92, 73), Array(93, 73), Array(94, 73), Array(95, 73), Array(96, 73), Array(103, 73), Array(104, 73), Array(105, 73), Array(106, 72), Array(108, 72), Array(110, 69), Array(111, 66), Array(110, 57), Array(108, 57), Array(105, 57), Array(95, 56), Array(84, 56), Array(81, 59), Array(81, 60), Array(83, 67), Array(90, 69), Array(97, 69), Array(104, 67), Array(106, 66), Array(108, 65), Array(108, 64), Array(117, 41), Array(118, 39), Array(118, 37), Array(121, 27), Array(121, 26), Array(121, 24), Array(121, 21), Array(118, 27), Array(118, 29), Array(118, 31), Array(118, 33), Array(118, 35), Array(118, 37), Array(118, 39), Array(118, 41), Array(118, 43), Array(120, 57), Array(128, 65), Array(134, 69), Array(138, 65), Array(137, 56), Array(137, 55), Array(133, 54), Array(128, 54), Array(126, 55), Array(125, 59), Array(125, 60), Array(125, 64), Array(125, 65), Array(127, 67), Array(128, 68), Array(130, 68), Array(139, 68), Array(140, 67), Array(146, 60), Array(146, 58), Array(147, 69), Array(148, 71), Array(152, 72), Array(154, 72), Array(155, 72), Array(163, 71), Array(164, 70), Array(166, 65), Array(167, 58), Array(167, 54), Array(167, 50), Array(166, 50), Array(161, 49), Array(149, 51), Array(148, 51), Array(147, 51), Array(146, 55), Array(146, 56), Array(146, 57), Array(147, 58), Array(147, 60), Array(149, 61), Array(149, 62), Array(149, 63) );
        $this->dtw->setReference($reference);
        // попытка 1 data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAASwAAACWCAYAAABkW7XSAAAFjUlEQVR4nO3dy47cRBQG4POw5DmAZ8oe2LAkKKwjVokiZZEICFFuM82iU1KN8aVcLrere75PshRNu33pif85ros7AgAAAAAAAAAAAAAAAACAS/kaEffZAtCl08QC0JVTRNyt+DnAIb7GdCV1FwIL6MhcFSWwgK7MtVMJLKAbdyGwgCux1KgusIBuLA1bEFhAF5ZuB9M6Ags4XMkYK4EFdKFkFLvAAg5XcjuY1hNYwKFKp9wILOBQc1NxhgQWcKg1ISSwgEMJLOBqCCzgaggs4GpodAeuyinKntkusIDDvQkDR4Erch/LoVVaiQHs7hQRHxZe/3yhYwGYNTen8N+Z1wAOcYpzz+FQ6QRpgIv5EuPBpP0K2NWXePhV82PLmLFwmqq8AKqknr61y9CPg5//OrHeXn6PczjexXLgqvrgCqXwuY9zZVVqaqR7PszhUu1Xv8TD89gSukCnThHxaeP7xwaEpu2WjNHaKgXPHxveD3SuRWP4VJWVKqvSJ5LWehvbx3ddIlSBqJ/y0vIinauyTnFux9rL22/LFn+HwOKROaoxtyawPkbbC3SqylrzNIdaLQLruxBYPCJHNubWBNYeE5HHqqxUxU0FeItjaBFYEefjfNZgO9C1qSpiahBlazXhs0ebzdjnkELsFOfhDsPX8l69Wi0Dy1gxbt5co/IluvRrAutzPOz+b2X4WZziPI9wbD+niHiaHUs6nhcr9/nnt2Uro/F5FJYu+L17oGpv7/Lqp5X8ok9BlL+WxniN7fdFPKy6Snv+WgXWJdrb4FClF/yef71rG/n3DqxhUOf7Wvo88oGfS1oF1rPC/cHVWvMffK/xSGkKylp5KKwZ3T4nP8exyi8PyY8F2yupHn+KiJ9XHOOcU5x7DOHm1NyKtQyHpPaWM90Ctaz+8obrsc/nQ5RXTlPbSNJ5fx8RP6w6ymmnOI/Jgpuzpe3or4bHsaU3MlU8rSq/fJrP1HZbBVaq1FoHloZ3blJPX6pQOx9wj8B6/e3fS4H1pmB7JYHVUk+/U2iqp7/Gte1j6RxaXKSv4/+9gu8H66QR9qW3sSWB9XzdYVbvD67a1qcctFQbnvkE5a3y6T5TAXo/WGdpoOZcgKTjbjnYU2Bxk1rPxduq9vYova/F7VVJI/5SL+LY8U3dVi5N+1nLc+e5Wb09jiRdbGsv3vS+4SDPLccwtZ2nI6+lY566LZ0LrPfR7vdwybmfcHG93Tqk4zlFxMuV78urlS3ntBQeU6+fRpb7mfX/yX7eoirqqS0SdtFrYK2tOPKgy8PiLs63eJ8i4lXhdpaqlNLPbGlb+Tk+X9jnnHexz7g46E6vgRWxrlLK3zcMrpplSb5uq4cO1vQUpu08qTgGuDo9B9aaDoGx8xhrT3r1bbvDb6OpqerSfsduAeeWqVBc21OovYpHp+fAiihvlxk7jzyQlrRo9C6t7Kae3lB6rk9WrAs3pffAijhfnEsTjJdGo8+FUS+VSsnk79QL+m7/w4H+XENglfSglTQ6j91y9VaplDbSw6PUW2BNBchcsKy5kPO2ptajy1saBnAvVSAcbq8Lt+aBfFO9bi9j+oKtbSjv/fErW3si4Wa1/gu+dphA/r7hRONkaihAT7d0rX2NiN+OPgjoUas5aFMN50veF6w3HDrgNgkesRYVy1iIlIypKv3yhJrKDbhRc2OFSt47pXaOHsCsmgqmJOjmKrjeeiyBK5I/pyktUwM41w4tGHtgoMACNsm/zXi41DZ+T7VVCSyguU/xcBJvTbvT2ERh7VdAt2rHagEAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAe/oPlVlDnFhEX2cAAAAASUVORK5CYII=
        $check = Array( Array(43, 55), Array(35, 46), Array(34, 46), Array(33, 46), Array(32, 46), Array(31, 46), Array(30, 48), Array(29, 51), Array(28, 52), Array(28, 53), Array(28, 55), Array(27, 59), Array(27, 63), Array(27, 64), Array(27, 67), Array(28, 68), Array(30, 69), Array(32, 72), Array(35, 74), Array(36, 74), Array(37, 74), Array(41, 75), Array(42, 75), Array(43, 75), Array(44, 75), Array(46, 72), Array(48, 70), Array(48, 69), Array(49, 68), Array(50, 68), Array(51, 68), Array(53, 70), Array(56, 74), Array(57, 76), Array(61, 76), Array(62, 75), Array(62, 74), Array(62, 72), Array(63, 66), Array(63, 63), Array(63, 61), Array(63, 58), Array(63, 57), Array(62, 50), Array(62, 44), Array(62, 45), Array(62, 46), Array(62, 49), Array(62, 51), Array(62, 52), Array(63, 57), Array(71, 70), Array(72, 70), Array(73, 70), Array(74, 70), Array(77, 65), Array(77, 48), Array(78, 53), Array(78, 55), Array(78, 56), Array(79, 57), Array(80, 60), Array(81, 62), Array(82, 62), Array(82, 64), Array(84, 64), Array(86, 66), Array(88, 67), Array(89, 67), Array(96, 61), Array(97, 49), Array(94, 43), Array(93, 41), Array(92, 40), Array(88, 40), Array(86, 40), Array(85, 40), Array(80, 41), Array(79, 41), Array(78, 42), Array(75, 46), Array(75, 47), Array(78, 60), Array(80, 61), Array(92, 62), Array(95, 61), Array(100, 55), Array(100, 51), Array(104, 34), Array(104, 27), Array(104, 16), Array(104, 15), Array(103, 13), Array(102, 13), Array(100, 13), Array(98, 14), Array(97, 16), Array(94, 25), Array(93, 27), Array(93, 29), Array(93, 31), Array(108, 57), Array(108, 58), Array(109, 59), Array(111, 59), Array(114, 60), Array(115, 60), Array(116, 60), Array(117, 59), Array(117, 58), Array(117, 57), Array(117, 56), Array(117, 55), Array(117, 54), Array(117, 51), Array(108, 47), Array(104, 60), Array(104, 61), Array(107, 62), Array(110, 63), Array(111, 63), Array(112, 63), Array(114, 63), Array(115, 63), Array(116, 63), Array(117, 63), Array(119, 63), Array(122, 61), Array(123, 57), Array(124, 56), Array(124, 54), Array(124, 53), Array(124, 52), Array(126, 64), Array(126, 65), Array(127, 65), Array(128, 66), Array(129, 67), Array(131, 68), Array(145, 57), Array(145, 56), Array(145, 54), Array(143, 43), Array(142, 42), Array(141, 41), Array(132, 41), Array(131, 41), Array(130, 41), Array(128, 42), Array(126, 43), Array(126, 48), Array(126, 51), Array(126, 52), Array(126, 54) );
        $this->dtw->setCheck($check);
        $minCost_1 = $this->dtw->getMinCost();
        // попытка 2(+) data:image/png;base64,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
        $check = Array( Array(64, 49), Array(63, 48), Array(62, 48), Array(60, 48), Array(57, 48), Array(56, 50), Array(55, 50), Array(53, 51), Array(51, 52), Array(49, 53), Array(42, 58), Array(38, 65), Array(38, 67), Array(37, 68), Array(37, 70), Array(37, 75), Array(37, 82), Array(40, 83), Array(64, 85), Array(85, 90), Array(85, 89), Array(85, 88), Array(85, 87), Array(86, 85), Array(86, 77), Array(87, 71), Array(87, 68), Array(88, 65), Array(88, 62), Array(89, 49), Array(89, 41), Array(89, 40), Array(90, 40), Array(91, 40), Array(93, 48), Array(93, 50), Array(94, 53), Array(94, 56), Array(96, 59), Array(97, 61), Array(101, 72), Array(104, 79), Array(104, 80), Array(104, 81), Array(106, 82), Array(106, 83), Array(107, 85), Array(114, 68), Array(122, 79), Array(123, 79), Array(124, 79), Array(125, 79), Array(126, 79), Array(127, 79), Array(127, 78), Array(131, 73), Array(137, 66), Array(137, 64), Array(137, 61), Array(137, 52), Array(134, 48), Array(130, 47), Array(127, 47), Array(124, 47), Array(120, 50), Array(120, 51), Array(120, 52), Array(118, 68), Array(118, 69), Array(119, 70), Array(120, 71), Array(149, 48), Array(149, 49), Array(149, 52), Array(150, 53), Array(151, 55), Array(155, 64), Array(155, 67), Array(157, 69), Array(158, 71), Array(158, 74), Array(158, 75), Array(158, 78), Array(158, 80), Array(158, 81), Array(158, 85), Array(158, 86), Array(158, 83), Array(148, 47), Array(148, 46), Array(153, 43), Array(156, 43), Array(157, 43), Array(158, 42), Array(160, 42), Array(163, 42), Array(164, 42), Array(165, 42), Array(166, 44), Array(167, 46), Array(168, 55), Array(168, 57), Array(167, 58), Array(164, 62), Array(151, 68), Array(152, 68), Array(161, 64), Array(163, 63), Array(175, 63), Array(176, 65), Array(177, 67), Array(178, 68), Array(178, 71), Array(178, 77), Array(177, 84), Array(175, 85), Array(171, 86), Array(165, 86), Array(164, 86), Array(153, 86), Array(152, 86), Array(151, 85), Array(191, 53), Array(191, 56), Array(191, 57), Array(191, 63), Array(191, 67), Array(192, 68), Array(192, 69), Array(195, 75), Array(199, 77), Array(200, 77), Array(201, 77), Array(203, 77), Array(209, 77), Array(210, 77), Array(211, 77), Array(213, 76), Array(216, 73), Array(218, 68), Array(218, 66), Array(218, 63), Array(218, 61), Array(215, 48), Array(214, 47), Array(214, 46), Array(213, 46), Array(210, 45), Array(202, 45), Array(201, 45), Array(200, 45), Array(198, 45), Array(193, 57), Array(193, 62) );
        $this->dtw->setCheck($check);
        $minCost_2 = $this->dtw->getMinCost();
        // Попытка 3 data:image/png;base64,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
        $check = Array( Array(38, 54), Array(36, 53), Array(36, 52), Array(32, 48), Array(32, 47), Array(31, 47), Array(30, 47), Array(30, 46), Array(29, 46), Array(28, 46), Array(28, 45), Array(29, 45), Array(30, 44), Array(31, 44), Array(32, 44), Array(33, 44), Array(34, 44), Array(37, 42), Array(38, 36), Array(38, 28), Array(38, 26), Array(38, 27), Array(38, 28), Array(38, 30), Array(39, 32), Array(39, 33), Array(41, 36), Array(41, 39), Array(42, 41), Array(42, 51), Array(45, 65), Array(46, 68), Array(47, 72), Array(53, 80), Array(54, 84), Array(55, 86), Array(55, 87), Array(55, 88), Array(54, 88), Array(53, 88), Array(42, 89), Array(37, 89), Array(34, 88), Array(33, 81), Array(34, 77), Array(40, 68), Array(40, 66), Array(42, 64), Array(43, 56), Array(45, 50), Array(46, 48), Array(46, 46), Array(46, 45), Array(46, 43), Array(46, 39), Array(46, 38), Array(50, 34), Array(51, 33), Array(52, 33), Array(55, 33), Array(56, 33), Array(57, 33), Array(58, 33), Array(61, 34), Array(63, 36), Array(63, 37), Array(62, 37), Array(61, 37), Array(61, 36), Array(58, 35), Array(57, 35), Array(56, 36), Array(54, 43), Array(52, 48), Array(49, 56), Array(48, 62), Array(46, 65), Array(43, 66), Array(42, 66), Array(42, 65), Array(43, 65), Array(45, 64), Array(46, 63), Array(47, 63), Array(50, 61), Array(53, 61), Array(54, 61), Array(57, 62), Array(59, 62), Array(59, 64), Array(59, 66), Array(59, 69), Array(59, 74), Array(59, 75), Array(66, 80), Array(68, 80), Array(73, 75), Array(73, 74), Array(73, 70), Array(73, 65), Array(74, 67), Array(75, 68), Array(75, 69), Array(75, 71), Array(77, 72), Array(78, 73), Array(79, 74), Array(80, 74), Array(81, 74), Array(82, 74), Array(83, 74), Array(85, 72), Array(85, 60), Array(83, 57), Array(82, 57), Array(78, 59), Array(77, 61), Array(76, 77), Array(78, 77), Array(86, 75), Array(88, 73), Array(94, 66), Array(96, 51), Array(96, 44), Array(95, 30), Array(94, 32), Array(91, 42), Array(91, 49), Array(91, 51), Array(95, 58), Array(113, 69), Array(113, 67), Array(113, 66), Array(114, 63), Array(100, 60), Array(99, 62), Array(99, 63), Array(111, 74), Array(112, 74), Array(113, 74), Array(120, 73), Array(126, 65), Array(129, 59), Array(130, 55), Array(130, 53), Array(130, 52), Array(128, 52), Array(125, 51), Array(124, 51), Array(123, 51), Array(122, 55), Array(124, 60), Array(129, 65), Array(132, 66), Array(133, 66), Array(134, 66), Array(135, 66), Array(137, 67), Array(142, 67), Array(142, 66), Array(144, 65), Array(148, 46), Array(148, 45), Array(148, 47), Array(149, 76), Array(149, 79), Array(149, 89), Array(149, 95), Array(149, 98), Array(149, 97), Array(133, 45), Array(133, 44), Array(134, 43), Array(136, 40), Array(140, 40), Array(145, 40), Array(159, 51), Array(160, 53), Array(157, 63), Array(156, 63), Array(155, 63), Array(153, 64), Array(152, 64) );
        $this->dtw->setCheck($check);
        $minCost_3 = $this->dtw->getMinCost();
        // Попытка 4 data:image/png;base64,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
        $check = Array( Array(125, 76), Array(126, 76), Array(127, 76), Array(127, 75), Array(130, 75), Array(137, 75), Array(142, 75), Array(145, 76), Array(151, 85), Array(151, 86), Array(151, 87), Array(151, 88), Array(151, 89), Array(151, 90), Array(151, 91), Array(151, 93), Array(149, 96), Array(148, 97), Array(142, 99), Array(141, 99), Array(137, 101), Array(133, 101), Array(127, 102), Array(125, 102), Array(118, 102), Array(102, 92), Array(101, 88), Array(99, 81), Array(99, 79), Array(99, 78), Array(99, 77), Array(99, 74), Array(100, 72), Array(105, 66), Array(110, 63), Array(114, 60), Array(120, 58), Array(125, 55), Array(137, 55), Array(141, 55), Array(148, 55), Array(156, 56), Array(158, 56), Array(166, 67), Array(167, 69), Array(168, 75), Array(168, 79), Array(168, 80), Array(168, 82), Array(168, 83), Array(167, 90), Array(166, 92), Array(165, 94), Array(164, 96), Array(164, 97), Array(163, 99), Array(156, 104), Array(150, 110), Array(141, 115), Array(130, 121), Array(127, 121), Array(120, 124), Array(106, 123), Array(100, 121), Array(83, 111), Array(74, 100), Array(72, 98), Array(72, 95), Array(71, 87), Array(91, 46), Array(101, 41), Array(103, 39), Array(111, 35), Array(120, 32), Array(139, 30), Array(143, 30), Array(154, 30), Array(156, 30), Array(160, 30), Array(166, 30), Array(173, 31), Array(188, 38), Array(193, 59), Array(193, 60), Array(193, 61), Array(193, 67), Array(193, 76), Array(179, 96), Array(179, 98), Array(176, 101), Array(171, 107), Array(164, 111), Array(147, 123), Array(145, 124), Array(144, 125), Array(142, 125), Array(141, 127), Array(134, 127), Array(129, 128), Array(123, 130), Array(116, 130), Array(105, 130), Array(96, 130), Array(82, 129), Array(71, 126), Array(69, 124), Array(51, 105), Array(47, 97), Array(46, 95), Array(45, 86), Array(45, 83), Array(52, 65), Array(58, 54), Array(60, 52), Array(75, 43), Array(88, 37), Array(111, 28), Array(131, 23), Array(156, 19), Array(159, 19), Array(170, 19), Array(177, 19), Array(211, 26), Array(222, 31), Array(228, 36), Array(229, 40), Array(219, 59), Array(217, 66), Array(217, 67), Array(235, 69), Array(246, 69), Array(256, 69), Array(259, 68) );
        $this->dtw->setCheck($check);
        $minCost_4 = $this->dtw->getMinCost();
        // Попытка 5 data:image/png;base64,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
        $check = Array( Array(53, 42), Array(53, 43), Array(54, 47), Array(55, 48), Array(56, 45), Array(56, 44), Array(56, 43), Array(56, 42), Array(56, 41), Array(56, 40), Array(54, 36), Array(53, 35), Array(52, 35), Array(51, 35), Array(50, 35), Array(46, 35), Array(44, 36), Array(40, 39), Array(38, 43), Array(37, 45), Array(37, 50), Array(37, 51), Array(37, 52), Array(37, 53), Array(37, 61), Array(37, 63), Array(40, 70), Array(46, 74), Array(48, 75), Array(49, 76), Array(52, 76), Array(53, 77), Array(54, 77), Array(54, 78), Array(56, 78), Array(57, 78), Array(58, 78), Array(61, 78), Array(62, 78), Array(66, 74), Array(67, 70), Array(70, 70), Array(71, 71), Array(81, 67), Array(81, 66), Array(81, 65), Array(81, 63), Array(81, 61), Array(81, 59), Array(81, 52), Array(81, 47), Array(81, 39), Array(81, 31), Array(81, 30), Array(81, 29), Array(81, 30), Array(81, 31), Array(81, 36), Array(81, 37), Array(83, 43), Array(83, 45), Array(92, 65), Array(93, 65), Array(94, 65), Array(97, 66), Array(101, 66), Array(102, 66), Array(105, 65), Array(107, 64), Array(108, 62), Array(109, 59), Array(111, 45), Array(111, 44), Array(111, 43), Array(111, 41), Array(111, 37), Array(115, 57), Array(116, 59), Array(125, 61), Array(127, 61), Array(133, 61), Array(134, 61), Array(135, 59), Array(136, 57), Array(137, 48), Array(137, 46), Array(128, 37), Array(123, 35), Array(114, 35), Array(110, 35), Array(107, 36), Array(105, 39), Array(104, 43), Array(104, 44), Array(104, 45), Array(104, 46), Array(104, 47), Array(105, 50), Array(106, 52), Array(107, 53), Array(110, 56), Array(111, 56), Array(114, 56), Array(116, 56), Array(117, 56), Array(120, 55), Array(122, 55), Array(123, 55), Array(124, 54), Array(125, 54), Array(126, 54), Array(132, 48), Array(135, 45), Array(136, 45), Array(137, 44), Array(138, 43), Array(139, 42), Array(142, 27), Array(142, 25), Array(143, 23), Array(143, 22), Array(143, 9), Array(143, 4), Array(140, 15), Array(140, 18), Array(140, 20), Array(140, 28), Array(140, 31), Array(141, 37), Array(143, 40), Array(144, 42), Array(145, 43), Array(146, 46), Array(147, 47), Array(150, 48), Array(150, 49), Array(151, 50), Array(152, 50), Array(153, 50), Array(158, 49), Array(158, 48), Array(158, 47), Array(158, 45), Array(158, 43), Array(158, 42), Array(157, 41), Array(156, 40), Array(155, 39), Array(152, 39), Array(151, 39), Array(148, 39), Array(147, 43), Array(147, 44), Array(147, 45), Array(149, 51), Array(150, 52), Array(164, 54), Array(165, 54), Array(166, 54), Array(168, 53), Array(169, 53), Array(169, 52), Array(171, 52), Array(172, 50), Array(173, 48), Array(173, 47), Array(174, 44), Array(174, 45), Array(174, 47), Array(174, 48), Array(174, 50), Array(174, 51), Array(175, 53), Array(177, 58), Array(178, 58), Array(179, 59), Array(180, 59), Array(181, 59), Array(182, 59), Array(183, 59), Array(184, 58), Array(188, 56), Array(191, 52), Array(192, 51), Array(192, 49), Array(192, 48), Array(192, 46), Array(186, 36), Array(184, 35), Array(179, 34), Array(177, 34), Array(176, 35), Array(175, 40), Array(175, 41), Array(175, 44) );
        $this->dtw->setCheck($check);
        $minCost_5 = $this->dtw->getMinCost();
        $this->assertTrue($minCost_1 == min($minCost_1, $minCost_2, $minCost_3, $minCost_4, $minCost_5));
    }


    public function provider() {
        return Array(
          Array( Array(1,0,2,03,3,1,3,2,34,200) ),
          Array( Array(1,0,2,03,3,2,34,200) ),
          Array( Array(1,0,2,03,3,1,3,2) ),
          Array( Array(1,0,2,03,3,1,3,2,200) )
        );
    }
}